﻿' 版权所有 (C) Microsoft Corporation。保留所有权利。
Option Infer On
Option Strict On

Imports SampleQueries.SampleSupport
Imports SampleQueries.NorthwindInheritance
Imports System.Collections.Generic
Imports System.Data
Imports System.Data.Linq
Imports System.Data.Linq.Mapping
Imports System.Data.Linq.SqlClient
Imports System.Data.SqlClient
Imports System.IO
Imports System.Linq
Imports System.Linq.Expressions
Imports System.Reflection
Imports System.Transactions
Imports System.Windows.Forms

<Title("LINQ to SQL 示例")> <Prefix("LinqToSql")> _
Public Class LinqToSQLSamples
    Inherits SampleHarness
    '连接字符串存储在 Sample Queries 应用程序设置中。要打开，请在 Sample Queries 项目的解决方案资源管理器中双击“我的项目”。 
    '如果在进行开发的计算机上安装了 SQL Server Express，则示例的默认连接字符串无需修改就可以正常使用。但是，在少数情况下可能会出现问题。
    ' 	在某些 SQL Express 安装上，有时可能没有权限启动用户实例。如果出现与此相关的错误消息，请尝试从连接字符串中移除子句“;user instance = true”。
    '	请确保您要访问的 Northwind 数据库副本没有标记为只读。如果需要，请浏览至要访问的 NORTHWIND.MDF 副本，右键单击该文件，选择“属性”，然后取消选中“只读”属性。
    '	在某些速度较慢的计算机上，或在使用虚拟机时，用户报告说一些数据库应用程序在首次运行时会出现错误。请尝试运行示例两到三次。
    '	如果使用的是 SQL Server 而不是 SQL Server Express，您需要更改连接字符串。如果您使用的不是 SQL Express 而是 SQL Server，可以根据自己的需要修改以下备用连接字符串。

    '   Server=[SERVER_NAME];Database=northwind;user id=[USER_NAME];password=[PASSWORD]

    '   不论 SERVER_NAME、USER_NAME 和 PASSWORD 出现在示例连接字符串中的什么地方，都有可能需要修改这些词。例如，您的连接字符串可能如下所示:

    '  Server=MyServer;Database=northwind;user id=sa;password=MyPassword"

    Private db As NorthwindDataContext
    Private newDB As NorthwindInheritance



    <Category("您的第一个 LINQ 查询")> _
    <Title("选择 2 列")> _
    <Description("此示例从数据库中选择 2 列并返回数据。")> _
    Public Sub LinqToSqlFirst01()

        '不是返回整个 Customers 表，而只是返回
        'CompanyName 和 Country
        Dim londonCustomers = From cust In db.Customers _
                              Select cust.CompanyName, cust.Country

        '执行查询并输出结果
        For Each custRow In londonCustomers
            Console.WriteLine("Company: " & custRow.CompanyName & vbTab & _
                              "Country: " & custRow.Country)
        Next
    End Sub

    <Category("您的第一个 LINQ 查询")> _
    <Title("简单筛选")> _
    <Description("此示例使用 Where 子句筛选在伦敦的客户。")> _
    Public Sub LinqToSqlFirst02()

        '仅返回伦敦的客户
        Dim londonCustomers = From cust In db.Customers _
                              Where cust.City = "London" _
                              Select cust.CompanyName, cust.City, cust.Country

        '执行查询并输出结果
        For Each custRow In londonCustomers
            Console.WriteLine("Company: " & custRow.CompanyName & vbTab & _
                              "City: " & custRow.City & vbTab & _
                              "Country: " & custRow.Country)
        Next
    End Sub


    <Category("WHERE")> _
    <Title("Where - 1")> _
    <Description("此示例使用 Where 子句筛选 " & _
                 "1994 年或之后雇用的雇员。")> _
    Public Sub LinqToSqlWhere01()
        Dim hiredAfter1994 = From emp In db.Employees _
                             Where emp.HireDate >= #1/1/1994# _
                             Select emp

        ObjectDumper.Write(hiredAfter1994)
    End Sub

    <Category("WHERE")> _
    <Title("Where - 2")> _
    <Description("此示例使用 Where 子句筛选库存量在订货点水平之下" & _
                 "但未断货的产品。")> _
    Public Sub LinqToSqlWhere02()
        Dim needToOrder = From prod In db.Products _
                          Where prod.UnitsInStock <= prod.ReorderLevel _
                                AndAlso Not prod.Discontinued _
                          Select prod

        ObjectDumper.Write(needToOrder)
    End Sub

    <Category("WHERE")> _
    <Title("Where - 3")> _
    <Description("此示例使用 Where 子句筛选" & _
                 "断货或单价大于 10 的产品。")> _
    Public Sub LinqToSqlWhere03()
        Dim prodQuery = From prod In db.Products _
                        Where prod.UnitPrice > 10.0# OrElse prod.Discontinued

        ObjectDumper.Write(prodQuery, 0)
    End Sub

    <Category("WHERE")> _
    <Title("Where - 4")> _
    <Description("此示例使用两个 Where 子句筛选断货" & _
                 "且单价大于 10 的产品")> _
    Public Sub LinqToSqlWhere04()

        Dim prodQuery = From prod In db.Products _
                        Where prod.UnitPrice > 10D _
                        Where prod.Discontinued

        ObjectDumper.Write(prodQuery, 0)
    End Sub


    <Category("WHERE")> _
    <Title("First - 简单用法")> _
    <Description("此示例使用 First 选择表中的第一个发货方。")> _
    Public Sub LinqToSqlWhere05()
        Dim shipper = db.Shippers.First()

        ObjectDumper.Write(shipper, 0)
    End Sub


    <Category("WHERE")> _
    <Title("First - 元素")> _
    <Description("此示例使用 Take 选择 CustomerID 为“BONAP”的第一个客户。")> _
    Public Sub LinqToSqlWhere06()
        Dim customer = From cust In db.Customers _
                       Where cust.CustomerID = "BONAP" _
                       Take 1

        ObjectDumper.Write(customer, 0)
    End Sub

    <Category("WHERE")> _
    <Title("First - 条件")> _
    <Description("此示例使用 First 选择运费大于 10.00 的订单。")> _
    Public Sub LinqToSqlWhere07()
        Dim firstOrd = (From ord In db.Orders _
                        Where ord.Freight > 10D _
                        Select ord).First()

        ObjectDumper.Write(firstOrd, 0)
    End Sub

    <Category("SELECT/DISTINCT")> _
    <Title("Select - 简单用法")> _
    <Description("此示例使用 Select 返回" & _
                 "仅含客户联系人姓名的序列。")> _
    Public Sub LinqToSqlSelect01()
        Dim contactList = From cust In db.Customers _
                          Select cust.ContactName

        ObjectDumper.Write(contactList)
    End Sub

    <Category("SELECT/DISTINCT")> _
    <Title("Select - 匿名类型 1")> _
    <Description("此示例使用 Select 和匿名类型返回 " & _
                 "仅含客户联系人姓名和电话号码的序列。")> _
    Public Sub LinqToSqlSelect02()
        Dim nameAndNumber = From cust In db.Customers _
                            Select cust.ContactName, cust.Phone

        ObjectDumper.Write(nameAndNumber)
    End Sub


    <Category("SELECT/DISTINCT")> _
    <Title("Select - 匿名类型 2")> _
    <Description("此示例使用 Select 和匿名类型返回" & _
                 "仅含雇员姓名和电话号码的序列，" & _
                 "并将 FirstName 和 LastName 字段合并为一个字段“Name”， " & _
                 "此外在所得的序列中将 HomePhone 字段重命名为 Phone。")> _
    Public Sub LinqToSqlSelect03()
        Dim nameAndNumber = From emp In db.Employees _
                            Select Name = emp.FirstName & " " & emp.LastName, _
                                   Phone = emp.HomePhone

        ObjectDumper.Write(nameAndNumber, 1)
    End Sub

    <Category("SELECT/DISTINCT")> _
    <Title("Select - 匿名类型 3")> _
    <Description("此示例使用 Select 和匿名类型返回" & _
                 "所有产品的 ID 以及 HalfPrice" & _
                 "(设置为产品单价除以 2 所得的值)" & _
                 "的序列。")> _
    Public Sub LinqToSqlSelect04()
        Dim prices = From prod In db.Products _
                     Select prod.ProductID, HalfPrice = prod.UnitPrice / 2

        ObjectDumper.Write(prices, 1)
    End Sub

    <Category("SELECT/DISTINCT")> _
    <Title("Select - 条件")> _
    <Description("此示例使用 Select 和条件语句返回产品名称" & _
                 " 和产品供货状态的序列。")> _
    Public Sub LinqToSqlSelect05()
        Dim inStock = From prod In db.Products _
                      Select prod.ProductName, _
                             Availability = If((prod.UnitsInStock - prod.UnitsOnOrder) < 0, _
                                               "Out Of Stock", _
                                               "In Stock")
        ObjectDumper.Write(inStock, 1)
    End Sub

    <Category("SELECT/DISTINCT")> _
    <Title("Select - 已命名类型")> _
    <Description("此示例使用 Select 和已知类型返回雇员姓名的序列。")> _
    Public Sub LinqToSqlSelect06()
        Dim names = From emp In db.Employees _
                    Select New Name With {.FirstName = emp.FirstName, _
                                          .LastName = emp.LastName}

        ObjectDumper.Write(names, 1)
    End Sub

    <Category("SELECT/DISTINCT")> _
    <Title("Select - 筛选")> _
    <Description("此示例使用 Select 和 Where 子句返回" & _
                 "仅含伦敦客户联系人姓名的序列。")> _
    Public Sub LinqToSqlSelect07()
        Dim londonNames = From cust In db.Customers _
                          Where cust.City = "London" _
                          Select cust.ContactName

        ObjectDumper.Write(londonNames)
    End Sub

    <Category("SELECT/DISTINCT")> _
    <Title("Select - 整形")> _
    <Description("此示例使用 Select 和匿名类型返回" & _
                 "有关客户的数据的整形子集。")> _
    Public Sub LinqToSqlSelect08()
        Dim customers = From cust In db.Customers _
                        Select cust.CustomerID, CompanyInfo = New With {cust.CompanyName, _
                                                                        cust.City, _
                                                                        cust.Country}, _
                                                ContactInfo = New With {cust.ContactName, _
                                                                        cust.ContactTitle}

        ObjectDumper.Write(customers, 1)
    End Sub

    <Category("SELECT/DISTINCT")> _
    <Title("Select - 嵌套")> _
    <Description("此示例使用嵌套查询返回" & _
                 "所有订单及其 OrderID 的序列、" & _
                 "打折订单中商品的子序列以及免送货" & _
                 "所省下的金额。")> _
    Public Sub LinqToSqlSelect09()
        Dim orders = From ord In db.Orders _
                     Select ord.OrderID, DiscountedProducts = (From od In ord.Order_Details _
                                                               Where od.Discount > 0.0), _
                                         FreeShippingDiscount = ord.Freight

        ObjectDumper.Write(orders, 1)
    End Sub

    '' 电话号码转换程序，可将电话号码转换为 
    '' 国际格式(基于其所在国家)。
    '' 对于 Northwind 数据库中的电话号码，
    '' 此示例仅支持 USA 和 UK 格式。
    Public Function PhoneNumberConverter(ByVal Country As String, ByVal Phone As String) As String
        Phone = Phone.Replace(" ", "").Replace(")", ")-")
        Select Case Country
            Case "USA"
                Return "1-" & Phone
            Case "UK"
                Return "44-" & Phone
            Case Else
                Return Phone
        End Select
    End Function

    <Category("SELECT/DISTINCT")> _
    <Title("Select - 本地方法调用 1")> _
    <Description("此示例使用对 " & _
                 "“PhoneNumberConverter”的本地方法调用来将电话号码转换为" & _
                 "国际格式。")> _
    Public Sub LinqToSqlLocalMethodCall01()

        Dim q = From c In db.Customers _
                Where c.Country = "UK" Or c.Country = "USA" _
                Select c.CustomerID, c.CompanyName, Phone = c.Phone, InternationalPhone = PhoneNumberConverter(c.Country, c.Phone)

        ObjectDumper.Write(q)
    End Sub

    <Category("SELECT/DISTINCT")> _
    <Title("Select - 本地方法调用 2")> _
    <Description("此示例使用本地方法调用来" & _
                 "将电话号码转换为国际格式" & _
                 "并创建 XDocument。")> _
    Public Sub LinqToSqlLocalMethodCall02()

        Dim doc = <Customers>
                      <%= From c In db.Customers _
                          Where c.Country = "UK" Or c.Country = "USA" _
                          Select <Customer CustomerID=<%= c.CustomerID %>
                                     CompanyName=<%= c.CompanyName %>
                                     InternationalPhone=<%= PhoneNumberConverter(c.Country, c.Phone) %>/> %>
                  </Customers>

        Console.WriteLine(doc.ToString())
    End Sub


    <Category("SELECT/DISTINCT")> _
    <Title("Distinct")> _
    <Description("此示例使用 Distinct 选择" & _
                 "有客户的唯一城市的序列。")> _
    Public Sub LinqToSqlSelect10()
        Dim cities = From cust In db.Customers _
                     Select cust.City _
                     Distinct

        ObjectDumper.Write(cities)
    End Sub

    <Category("COUNT/SUM/MIN/MAX/AVG")> _
    <Title("Count - 简单用法")> _
    <Description("此示例使用 Count 得到数据库中客户的数量。")> _
    Public Sub LinqToSqlCount01()
        Dim custCount = db.Customers.Count()
        Console.WriteLine(custCount)
    End Sub

    <Category("COUNT/SUM/MIN/MAX/AVG")> _
    <Title("Count - 条件")> _
    <Description("此示例使用 Count 得到数据库中未断货产品" & _
                 "的数量。")> _
    Public Sub LinqToSqlCount02()
        Dim activeProducts = Aggregate prod In db.Products _
                             Into Count(Not prod.Discontinued)

        '可选语法
        'Dim activeProducts = (From prod In db.Products _
        '                      Where Not prod.Discontinued _
        '                      Select prod).Count()

        Console.WriteLine(activeProducts)
    End Sub

    <Category("COUNT/SUM/MIN/MAX/AVG")> _
    <Title("Sum - 简单用法")> _
    <Description("此示例使用 Sum 得到所有订单的总运费。")> _
    Public Sub LinqToSqlCount03()

        Dim totalFreight = Aggregate ord In db.Orders _
                           Into Sum(ord.Freight)

        '可选语法
        'Dim totalFreight = (From ord In db.Orders _
        '                    Select ord.Freight).Sum()

        Console.WriteLine(totalFreight)
    End Sub

    <Category("COUNT/SUM/MIN/MAX/AVG")> _
    <Title("Sum - 映射")> _
    <Description("此示例使用 Sum 得到所有产品的订货总数。")> _
    Public Sub LinqToSqlCount04()
        Dim totalUnits = (From prod In db.Products _
                          Select CInt(prod.UnitsOnOrder.Value)).Sum()

        Console.WriteLine(totalUnits)
    End Sub

    <Category("COUNT/SUM/MIN/MAX/AVG")> _
    <Title("Min - 简单用法")> _
    <Description("此示例使用 Min 查找任意产品的最低单价。")> _
    Public Sub LinqToSqlCount05()
        Dim lowestPrice = Aggregate prod In db.Products _
                          Into Min(prod.UnitPrice)

        Console.WriteLine(lowestPrice)
    End Sub

    <Category("COUNT/SUM/MIN/MAX/AVG")> _
    <Title("Min - 映射")> _
    <Description("此示例使用 Min 查找任意订单的最低运费。")> _
    Public Sub LinqToSqlCount06()
        Dim lowestFreight = Aggregate ord In db.Orders _
                            Into Min(ord.Freight)

        Console.WriteLine(lowestFreight)
    End Sub

    <Category("COUNT/SUM/MIN/MAX/AVG")> _
    <Title("Min - 元素")> _
    <Description("此示例使用 Min 查找每个类别中单价最低的" & _
                 "产品。")> _
    Public Sub LinqToSqlCount07()
        Dim categories = From prod In db.Products _
                         Group prod By prod.CategoryID Into g = Group _
                         Select CategoryID = g, _
                                CheapestProducts = _
                                    From p2 In g _
                                    Where p2.UnitPrice = g.Min(Function(p3) p3.UnitPrice) _
                                    Select p2

        ObjectDumper.Write(categories, 1)
    End Sub


    <Category("COUNT/SUM/MIN/MAX/AVG")> _
    <Title("Max - 简单用法")> _
    <Description("此示例使用 Max 查找任意雇员的最近雇用日期。")> _
    Public Sub LinqToSqlCount08()
        Dim latestHire = Aggregate emp In db.Employees _
                         Into Max(emp.HireDate)

        Console.WriteLine(latestHire)
    End Sub

    <Category("COUNT/SUM/MIN/MAX/AVG")> _
    <Title("Max - 映射")> _
    <Description("此示例使用 Max 查找任意产品的最大库存量。")> _
    Public Sub LinqToSqlCount09()
        Dim mostInStock = Aggregate prod In db.Products _
                          Into Max(prod.UnitsInStock)

        Console.WriteLine(mostInStock)
    End Sub

    <Category("COUNT/SUM/MIN/MAX/AVG")> _
    <Title("Max - 元素")> _
    <Description("此示例使用 Max 查找每个类别中单价最高的" & _
                 "产品。")> _
    Public Sub LinqToSqlCount10()
        Dim categories = From prod In db.Products _
                         Group prod By prod.CategoryID Into g = Group _
                         Select CategoryGroup = g, _
                                MostExpensiveProducts = _
                                    From p2 In g _
                                    Where p2.UnitPrice = g.Max(Function(p3) p3.UnitPrice)

        ObjectDumper.Write(categories, 1)
    End Sub


    <Category("COUNT/SUM/MIN/MAX/AVG")> _
    <Title("Average - 简单用法")> _
    <Description("此示例使用 Average 得到所有订单的平均运费。")> _
    Public Sub LinqToSqlCount11()
        Dim avgFreight = Aggregate ord In db.Orders _
                         Into Average(ord.Freight)

        Console.WriteLine(avgFreight)
    End Sub

    <Category("COUNT/SUM/MIN/MAX/AVG")> _
    <Title("Average - 映射")> _
    <Description("此示例使用 Average 得到所有产品的平均单价。")> _
    Public Sub LinqToSqlCount12()
        Dim avgPrice = Aggregate prod In db.Products _
                       Into Average(prod.UnitPrice)

        Console.WriteLine(avgPrice)
    End Sub

    <Category("COUNT/SUM/MIN/MAX/AVG")> _
    <Title("Average - 元素")> _
    <Description("此示例使用 Average 查找每个类别中单价高于" & _
                 "该类别平均单价的产品。")> _
    Public Sub LinqToSqlCount13()
        Dim categories = From prod In db.Products _
                         Group prod By prod.CategoryID Into g = Group _
                         Select g, _
                                ExpensiveProducts = _
                                    From prod2 In g _
                                    Where (prod2.UnitPrice > g.Average(Function(p3) p3.UnitPrice))

        ObjectDumper.Write(categories, 1)
    End Sub


    <Category("JOIN")> _
    <Title("SelectMany - 一对多 - 1")> _
    <Description("此示例在 From 子句中使用外键导航" & _
                 "选择伦敦客户的所有订单。")> _
    Public Sub LinqToSqlJoin01()
        Dim ordersInLondon = From cust In db.Customers, ord In cust.Orders _
                             Where cust.City = "London"

        ObjectDumper.Write(ordersInLondon, 1)
    End Sub

    <Category("JOIN")> _
    <Title("SelectMany - 一对多 - 2")> _
    <Description("此示例在 Where 子句中使用外键导航" & _
                 "筛选其供应商在美国且" & _
                 "缺货的产品。")> _
    Public Sub LinqToSqlJoin02()
        Dim outOfStock = From prod In db.Products _
                         Where prod.Supplier.Country = "USA" AndAlso prod.UnitsInStock = 0

        ObjectDumper.Write(outOfStock)
    End Sub

    <Category("JOIN")> _
    <Title("SelectMany - 多对多")> _
    <Description("此示例在 From 子句中使用外键导航" & _
                 "筛选在西雅图的雇员，" & _
                 "同时列出其所在地区。")> _
    Public Sub LinqToSqlJoin03()
        Dim seattleEmployees = From emp In db.Employees, et In emp.EmployeeTerritories _
                               Where emp.City = "Seattle" _
                               Select emp.FirstName, emp.LastName, et.Territory.TerritoryDescription

        ObjectDumper.Write(seattleEmployees)
    End Sub

    <Category("JOIN")> _
    <Title("SelectMany - 自联接")> _
    <Description("此示例在 Select 子句中使用外键导航" & _
                 "筛选成对的雇员，每对中" & _
                 "一个雇员隶属于另一个雇员，且" & _
                 "两个雇员都来自相同城市。")> _
    Public Sub LinqToSqlJoin04()
        Dim empQuery = From emp1 In db.Employees, emp2 In emp1.Employees _
                       Where emp1.City = emp2.City _
                       Select FirstName1 = emp1.FirstName, LastName1 = emp1.LastName, _
                              FirstName2 = emp2.FirstName, LastName2 = emp2.LastName, emp1.City

        ObjectDumper.Write(empQuery)
    End Sub

    <Category("JOIN")> _
    <Title("GroupJoin - 双向联接")> _
    <Description("此示例显式联接两个表并从这两个表投影出结果。")> _
    Public Sub LinqToSqlJoin05()
        Dim ordCount = From cust In db.Customers _
                       Group Join ord In db.Orders On cust.CustomerID Equals ord.CustomerID _
                       Into orders = Group _
                       Select cust.ContactName, OrderCount = orders.Count()

        ObjectDumper.Write(ordCount)
    End Sub
    <Category("JOIN")> _
    <Title("GroupJoin - 三向联接")> _
    <Description("此示例显式联接三个表并分别从每个表投影出结果。")> _
    Public Sub LinqToSqlJoin06()
        Dim joinQuery = From cust In db.Customers _
                        Group Join ord In db.Orders On cust.CustomerID Equals ord.CustomerID _
                              Into ords = Group _
                        Group Join emp In db.Employees On cust.City Equals emp.City _
                              Into emps = Group _
                        Select cust.ContactName, ords = ords.Count(), emps = emps.Count()

        ObjectDumper.Write(joinQuery)
    End Sub

    <Category("JOIN")> _
    <Title("GroupJoin - 左外部联接")> _
    <Description("此示例显示如何使用 DefaultIfEmpty() 获取左外部联接。" & _
                 "雇员没有订单时，DefaultIfEmpty() 方法返回 Nothing。")> _
    Public Sub LinqToSqlJoin07()
        Dim empQuery = From emp In db.Employees _
                       Group Join ord In db.Orders On emp Equals ord.Employee _
                             Into ords = Group _
                       From ord2 In ords.DefaultIfEmpty _
                       Select emp.FirstName, emp.LastName, Order = ord2

        ObjectDumper.Write(empQuery, 1)
    End Sub

    <Category("JOIN")> _
    <Title("GroupJoin - 投影的 Let 赋值")> _
    <Description("此示例从联接投影出最终“Let”表达式。")> _
    Public Sub LinqToSqlJoin08()
        Dim ordQuery = From cust In db.Customers _
                       Group Join ord In db.Orders On cust.CustomerID Equals ord.CustomerID _
                       Into ords = Group _
                       Let Location = cust.City + cust.Country _
                       From ord2 In ords _
                       Select cust.ContactName, ord2.OrderID, Location

        ObjectDumper.Write(ordQuery)
    End Sub

    <Category("JOIN")> _
    <Title("GroupJoin - 组合键")> _
    <Description("此示例显示带有组合键的联接。")> _
    Public Sub LinqToSqlJoin09()

        'Key 关键字表示在测试匿名类型是否相等时，
        '将仅比较 OrderID 字段
        Dim ordQuery = From ord In db.Orders _
                       From prod In db.Products _
                       Group Join details In db.Order_Details _
                           On New With {Key ord.OrderID, prod.ProductID} _
                           Equals New With {Key details.OrderID, details.ProductID} _
                       Into details = Group _
                       From d In details _
                       Select ord.OrderID, prod.ProductID, d.UnitPrice

        ObjectDumper.Write(ordQuery)
    End Sub

    <Category("JOIN")> _
    <Title("GroupJoin - 可为 null\\不可为 null 的键关系")> _
    <Description("此示例显示如何构造一侧可为 null 而另一侧不可为 null 的联接。")> _
    Public Sub LinqToSqlJoin10()
        Dim ordQuery = From ord In db.Orders _
                       Group Join emp In db.Employees _
                           On ord.EmployeeID Equals CType(emp.EmployeeID, Integer?) _
                       Into emps = Group _
                       From emp2 In emps _
                       Select ord.OrderID, emp2.FirstName

        ObjectDumper.Write(ordQuery)
    End Sub

    <Category("ORDER BY")> _
    <Title("OrderBy - 简单用法")> _
    <Description("此示例使用 Order By 按雇用日期对雇员" & _
                 "进行排序。")> _
    Public Sub LinqToSqlOrderBy01()
        Dim empQuery = From emp In db.Employees _
                       Order By emp.HireDate

        ObjectDumper.Write(empQuery)
    End Sub

    <Category("ORDER BY")> _
    <Title("OrderBy - 带 Where")> _
    <Description("此示例使用 Where 和 Order By 按运费对到货地为伦敦的订单" & _
                 "进行排序。")> _
    Public Sub LinqToSqlOrderBy02()
        Dim londonOrders = From ord In db.Orders _
                           Where ord.ShipCity = "London" _
                           Order By ord.Freight

        ObjectDumper.Write(londonOrders)
    End Sub

    <Category("ORDER BY")> _
    <Title("OrderByDescending")> _
    <Description("此示例使用 Order By 按单价从高到低对产品" & _
                 "进行排序。")> _
    Public Sub LinqToSqlOrderBy03()
        Dim sortedProducts = From prod In db.Products _
                             Order By prod.UnitPrice Descending

        ObjectDumper.Write(sortedProducts)
    End Sub

    <Category("ORDER BY")> _
    <Title("ThenBy")> _
    <Description("此示例使用复合 Order 按城市，再按联系人姓名对客户" & _
                 "进行排序。")> _
    Public Sub LinqToSqlOrderBy04()
        Dim custQuery = From cust In db.Customers _
                        Select cust _
                        Order By cust.City, cust.ContactName

        ObjectDumper.Write(custQuery)
    End Sub

    <Category("ORDER BY")> _
    <Title("ThenByDescending")> _
    <Description("此示例使用 Order By 按到货国家/地区，再按运费从高到低" & _
                 "对 EmployeeID 1 的订单进行排序。")> _
    Public Sub LinqToSqlOrderBy05()
        Dim ordQuery = From ord In db.Orders _
                       Where ord.EmployeeID = 1 _
                       Order By ord.ShipCountry, ord.Freight Descending

        ObjectDumper.Write(ordQuery)
    End Sub

    <Category("ORDER BY")> _
    <Title("OrderBy - Group By")> _
    <Description("此示例使用 Order By、Max 和 Group By 查找每个类别中单价最高的产品，" & _
                 "并按 CategoryID 对这组产品进行排序。")> _
    Public Sub LinqToSqlOrderBy06()
        Dim categories = From prod In db.Products _
                         Group prod By prod.CategoryID Into Group _
                         Order By CategoryID _
                         Select Group, _
                                MostExpensiveProducts = _
                                    From prod2 In Group _
                                    Where prod2.UnitPrice = _
                                        Group.Max(Function(prod3) prod3.UnitPrice)

        ObjectDumper.Write(categories, 1)
    End Sub

    <Category("GROUP BY/HAVING")> _
    <Title("GroupBy - 简单用法")> _
    <Description("此示例使用 Group By 按 CategoryID 划分" & _
                 "产品。")> _
    Public Sub LinqToSqlGroupBy01()
        Dim categorizedProducts = From prod In db.Products _
                                  Group prod By prod.CategoryID Into prodGroup = Group _
                                  Select prodGroup

        ObjectDumper.Write(categorizedProducts, 1)
    End Sub

    <Category("GROUP BY/HAVING")> _
    <Title("GroupBy - Max")> _
    <Description("此示例使用 Group By 和 Max " & _
                 "查找每个 CategoryID 的最高单价。")> _
    Public Sub LinqToSqlGroupBy02()
        Dim maxPrices = From prod In db.Products _
                        Group prod By prod.CategoryID _
                        Into prodGroup = Group, MaxPrice = Max(prod.UnitPrice) _
                        Select prodGroup, MaxPrice

        ObjectDumper.Write(maxPrices, 1)
    End Sub

    <Category("GROUP BY/HAVING")> _
    <Title("GroupBy - Min")> _
    <Description("此示例使用 Group By 和 Min " & _
                 "查找每个 CategoryID 的最低单价。")> _
    Public Sub LinqToSqlGroupBy03()
        Dim minPrices = From prod In db.Products _
                        Group prod By prod.CategoryID _
                        Into prodGroup = Group, MinPrice = Min(prod.UnitPrice)

        ObjectDumper.Write(minPrices, 1)
    End Sub

    <Category("GROUP BY/HAVING")> _
    <Title("GroupBy - Average")> _
    <Description("此示例使用 Group By 和 Average " & _
                 "得到每个 CategoryID 的平均单价。")> _
    Public Sub LinqToSqlGroupBy04()
        Dim avgPrices = From prod In db.Products _
                        Group prod By prod.CategoryID _
                        Into prodGroup = Group, AveragePrice = Average(prod.UnitPrice)

        ObjectDumper.Write(avgPrices, 1)
    End Sub

    <Category("GROUP BY/HAVING")> _
    <Title("GroupBy - Sum")> _
    <Description("此示例使用 Group By 和 Sum " & _
                 "得到每个 CategoryID 的单价总计。")> _
    Public Sub LinqToSqlGroupBy05()
        Dim totalPrices = From prod In db.Products _
                          Group prod By prod.CategoryID _
                          Into prodGroup = Group, TotalPrice = Sum(prod.UnitPrice)

        ObjectDumper.Write(totalPrices, 1)
    End Sub

    <Category("GROUP BY/HAVING")> _
    <Title("GroupBy - Count")> _
    <Description("此示例使用 Group By 和 Count " & _
                 "得到每个 CategoryID 中产品的数量。")> _
    Public Sub LinqToSqlGroupBy06()
        Dim prodQuery = From prod In db.Products _
                        Group prod By prod.CategoryID _
                        Into prodGroup = Group _
                        Select prodGroup, NumProducts = prodGroup.Count()

        ObjectDumper.Write(prodQuery, 1)
    End Sub

    <Category("GROUP BY/HAVING")> _
    <Title("GroupBy - Count - 条件")> _
    <Description("此示例使用 Group By 和 Count " & _
                 "得到每个 CategoryID 中断货" & _
                 "产品的数量。")> _
    Public Sub LinqToSqlGroupBy07()

        Dim prodQuery = From prod In db.Products _
                        Group prod By prod.CategoryID _
                        Into prodGroup = Group, NumProducts = Count(prod.Discontinued)

        '可选语法
        'Dim prodQuery = From prod In db.Products _
        '                Group prod By prod.CategoryID Into prodGroup = Group _
        '                Select prodGroup, _
        '                       NumProducts = prodGroup.Count(Function(prod2) prod2.Discontinued)

        ObjectDumper.Write(prodQuery, 1)
    End Sub

    <Category("GROUP BY/HAVING")> _
    <Title("GroupBy - 后跟 Where")> _
    <Description("此示例在 Group By 子句后使用 Where 子句" & _
                 "查找所有至少有 10 种产品的类别。")> _
    Public Sub LinqToSqlGroupBy08()
        Dim bigCategories = From prod In db.Products _
                            Group prod By prod.CategoryID _
                            Into ProdGroup = Group, ProdCount = Count() _
                            Where ProdCount >= 10 _
                            Select ProdGroup, ProdCount

        ObjectDumper.Write(bigCategories, 1)
    End Sub

    <Category("GROUP BY/HAVING")> _
    <Title("GroupBy - 多列")> _
    <Description("此示例使用 Group By 按 CategoryID 和 SupplierID 将产品分组。")> _
    Public Sub LinqToSqlGroupBy09()
        Dim categories = From prod In db.Products _
                         Group By Key = New With {prod.CategoryID, prod.SupplierID} _
                         Into prodGroup = Group _
                         Select Key, prodGroup

        ObjectDumper.Write(categories, 1)
    End Sub

    <Category("GROUP BY/HAVING")> _
    <Title("GroupBy - 表达式")> _
    <Description("此示例使用 Group By 返回两个产品序列。" & _
                 "第一个序列包含" & _
                 "单价大于 10 的产品。第二个序列包含" & _
                 "单价小于或等于 10 的产品。")> _
    Public Sub LinqToSqlGroupBy10()
        Dim categories = From prod In db.Products _
                         Group prod By Key = New With {.Criterion = prod.UnitPrice > 10} _
                         Into ProductGroup = Group

        ObjectDumper.Write(categories, 1)
    End Sub

    <Category("EXISTS/IN/ANY/ALL")> _
    <Title("Any - 简单用法")> _
    <Description("此示例使用 Any 运算符仅返回未下订单的客户。")> _
    Public Sub LinqToSqlExists01()
        Dim custQuery = From cust In db.Customers _
                        Where Not cust.Orders.Any()

        ObjectDumper.Write(custQuery)
    End Sub

    <Category("EXISTS/IN/ANY/ALL")> _
    <Title("Any - 条件")> _
    <Description("此示例使用 Any 仅返回至少有" & _
                 "一种产品断货的类别。")> _
    Public Sub LinqToSqlExists02()
        Dim prodQuery = From cust In db.Categories _
                        Where (From prod In cust.Products Where prod.Discontinued).Any()

        ObjectDumper.Write(prodQuery)
    End Sub

    <Category("EXISTS/IN/ANY/ALL")> _
    <Title("All - 条件")> _
    <Description("此示例使用 All 返回所有订单都运往" & _
                 "其所在城市的客户或未下订单的客户。")> _
    Public Sub LinqToSqlExists03()
        Dim ordQuery = From cust In db.Customers _
                       Where cust.Orders.All(Function(ord) ord.ShipCity = cust.City)

        ObjectDumper.Write(ordQuery)
    End Sub

    <Category("Exists/In/Any/All/Contains")> _
    <Title("Contains - 一个对象")> _
    <Description("此示例使用 Contain 来查找哪位客户包含 OrderID 为 10248 的订单。")> _
    Public Sub LinqToSqlExists04()

        Dim order = (From o In db.Orders _
                     Where o.OrderID = 10248).First()

        Dim q = db.Customers.Where(Function(p) p.Orders.Contains(order)).ToList()

        For Each cust In q
            For Each ord In cust.Orders

                Console.WriteLine("客户 {0} 具有 OrderID {1}。", _
                                  cust.CustomerID, ord.OrderID)
            Next
        Next

    End Sub

    <Category("Exists/In/Any/All/Contains")> _
    <Title("Contains - 多个值")> _
    <Description("此示例使用 Contains 查找所在城市为西雅图、伦敦、巴黎或范库弗峰的客户。")> _
    Public Sub LinqToSqlExists05()
        Dim cities = New String() {"Seattle", "London", "Vancouver", "Paris"}

        Dim q = db.Customers.Where(Function(p) cities.Contains(p.City)).ToList()

        ObjectDumper.Write(q)
    End Sub

    <Category("UNION ALL/UNION/INTERSECT")> _
    <Title("Concat - 简单用法")> _
    <Description("此示例使用 Concat 返回所有客户和雇员的" & _
                 "电话/传真号码的序列。")> _
    Public Sub LinqToSqlUnion01()
        Dim phoneNumbers = (From cust In db.Customers Select cust.Phone).Concat( _
                            From cust In db.Customers Select cust.Fax).Concat( _
                            From emp In db.Employees Select emp.HomePhone)

        ObjectDumper.Write(phoneNumbers)
    End Sub

    <Category("UNION ALL/UNION/INTERSECT")> _
    <Title("Concat - 复合")> _
    <Description("此示例使用 Concat 返回所有客户和雇员的" & _
                 "姓名与电话号码映射的序列。")> _
    Public Sub LinqToSqlUnion02()
        Dim custPhones = From cust In db.Customers _
                         Select Name = cust.CompanyName, _
                                Phone = cust.Phone

        Dim phoneNumbers = custPhones.Concat(From emp In db.Employees _
                                             Select Name = emp.FirstName & " " & emp.LastName, _
                                                    Phone = emp.HomePhone)

        ObjectDumper.Write(phoneNumbers)
    End Sub

    <Category("UNION ALL/UNION/INTERSECT")> _
    <Title("Union")> _
    <Description("此示例使用 Union 返回有客户或雇员任意一方居住的" & _
                 "所有国家/地区的序列。")> _
    Public Sub LinqToSqlUnion03()
        Dim countries = (From cust In db.Customers _
                         Select cust.Country).Union(From emp In db.Employees _
                                                    Select emp.Country)

        ObjectDumper.Write(countries)
    End Sub

    <Category("UNION ALL/UNION/INTERSECT")> _
    <Title("Intersect")> _
    <Description("此示例使用 Intersect 返回客户和雇员双方都居住的" & _
                 "所有国家/地区的序列。")> _
    Public Sub LinqToSqlUnion04()
        Dim countries = (From cust In db.Customers _
                         Select cust.Country).Intersect(From emp In db.Employees _
                                                        Select emp.Country)

        ObjectDumper.Write(countries)
    End Sub

    <Category("UNION ALL/UNION/INTERSECT")> _
    <Title("Except")> _
    <Description("此示例使用 Except 返回有客户居住但无雇员居住的" & _
                 "所有国家/地区的序列。")> _
    Public Sub LinqToSqlUnion05()
        Dim countries = (From cust In db.Customers _
                         Select cust.Country).Except(From emp In db.Employees _
                                                     Select emp.Country)

        ObjectDumper.Write(countries)
    End Sub

    <Category("TOP/BOTTOM")> _
    <Title("Take")> _
    <Description("此示例使用 Take 选择所雇用的前 5 个雇员。")> _
    Public Sub LinqToSqlTop01()
        Dim first5Employees = From emp In db.Employees _
                              Order By emp.HireDate _
                              Take 5

        ObjectDumper.Write(first5Employees)
    End Sub

    <Category("TOP/BOTTOM")> _
    <Title("Skip")> _
    <Description("此示例使用 Skip 选择 10 种最贵产品之外的所有产品。")> _
    Public Sub LinqToSqlTop02()
        Dim expensiveProducts = From prod In db.Products _
                                Order By prod.UnitPrice Descending _
                                Skip 10

        ObjectDumper.Write(expensiveProducts)
    End Sub

    <Category("分页")> _
    <Title("分页 - 索引")> _
    <Description("此示例使用 Skip 和 Take 运算符进行分页，" & _
                 "跳过前 50 个记录，然后返回接下来 10 个，因此" & _
                 "提供 Products 表第 6 页的数据。")> _
    Public Sub LinqToSqlPaging01()
        Dim productPage = From cust In db.Customers _
                          Order By cust.ContactName _
                          Skip 50 _
                          Take 10

        ObjectDumper.Write(productPage)
    End Sub

    <Category("分页")> _
    <Title("分页 - 按唯一键排序")> _
    <Description("此示例使用 Where 子句和 Take 运算符进行分页，" & _
                 "首先筛选得到仅 50 (第 5 页最后一个 ProductID " & _
                 ")以上的 ProductID，然后按 ProductID 排序，最后取前 10 个结果，" & _
                 "因此提供 Products 表第 6 页的数据。" & _
                 "请注意，此方法仅适用于按唯一键排序的情况。")> _
    Public Sub LinqToSqlPaging02()
        Dim productPage = From prod In db.Products _
                          Where prod.ProductID > 50 _
                          Select prod _
                          Order By prod.ProductID _
                          Take 10

        ObjectDumper.Write(productPage)
    End Sub

    <Category("SqlMethods")> _
    <Title("SqlMethods - Like")> _
    <Description("此示例使用 SqlMethods 筛选 CustomerID 以“C”开头的客户。")> _
    Public Sub LinqToSqlSqlMethods01()


        Dim q = From c In db.Customers _
                Where SqlMethods.Like(c.CustomerID, "C%") _
                Select c

        ObjectDumper.Write(q)

    End Sub

    <Category("SqlMethods")> _
    <Title("SqlMethods - DateDiffDay")> _
    <Description("此示例使用 SqlMethods 查找在创建定单后的 10 日内发货的所有定单")> _
    Public Sub LinqToSqlSqlMethods02()

        Dim orderQuery = From o In db.Orders _
                         Where SqlMethods.DateDiffDay(o.OrderDate, o.ShippedDate) < 10

        ObjectDumper.Write(orderQuery)
    End Sub


    <Category("已编译查询")> _
    <Title("已编译查询 - 1")> _
    <Description("此示例创建一个已编译查询，然后使用它检索位于所输入城市的客户")> _
    Public Sub LinqToSqlCompileQuery01()

        '' 创建已编译查询
        Dim fn = CompiledQuery.Compile( _
                Function(db2 As NorthwindDataContext, city As String) _
                    From c In db2.Customers _
                    Where c.City = city _
                    Select c)


        Console.WriteLine("****** Call compiled query to retrieve customers from London ******")
        Dim LonCusts = fn(db, "London")
        ObjectDumper.Write(LonCusts)

        Console.WriteLine()

        Console.WriteLine("****** Call compiled query to retrieve customers from Seattle ******")
        Dim SeaCusts = fn(db, "Seattle")
        ObjectDumper.Write(SeaCusts)

    End Sub


    <Category("INSERT/UPDATE/DELETE")> _
    <Title("插入 - 简单用法")> _
    <Description("此示例使用 Add 方法将新 Customer 添加到 " & _
                 "Customers 表对象。调用 SubmitChanges 将此" & _
                 "新 Customer 保持到数据库。")> _
    Public Sub LinqToSqlInsert01()
        Dim custQuery = From cust In db.Customers _
                        Where cust.Region = "WA"

        Console.WriteLine("*** BEFORE ***")
        ObjectDumper.Write(custQuery)


        Console.WriteLine()
        Console.WriteLine("*** INSERT ***")
        Dim newCustomer = New Customer With {.CustomerID = "MCSFT", .CompanyName = "Microsoft", .ContactName = "John Doe", .ContactTitle = "Sales Manager", .Address = "1 Microsoft Way", .City = "Redmond", .Region = "WA", .PostalCode = "98052", .Country = "USA", .Phone = "(425) 555-1234", .Fax = Nothing}
        db.Customers.InsertOnSubmit(newCustomer)
        db.SubmitChanges()


        Console.WriteLine()
        Console.WriteLine("*** AFTER ***")
        ObjectDumper.Write(custQuery)

        Cleanup56()  ' 还原以前的数据库状态
    End Sub

    Private Sub Cleanup56()
        setLogging(False)

        db.Customers.DeleteAllOnSubmit(From cust In db.Customers _
                                       Where cust.CustomerID = "MCSFT")
        db.SubmitChanges()
    End Sub

    <Category("INSERT/UPDATE/DELETE")> _
    <Title("Insert - 一对多")> _
    <Description("此示例使用 Add 方法将新 Category 添加到 " & _
                 "Categories 表对象，并将新 Product 添加到与此新 Category 有外键关系的 " & _
                 "Products 表。调用" & _
                 "SubmitChanges 将这些新对象及其关系保持到" & _
                 "数据库。")> _
    Public Sub LinqToSqlInsert02()

        Dim db2 = New NorthwindDataContext(My.Settings.NORTHWINDConnectionString1)

        Dim ds = New DataLoadOptions()

        ds.LoadWith(Of Category)(Function(p) p.Products)
        db2.LoadOptions = ds

        Dim q = From category In db2.Categories _
                Where category.CategoryName = "Widgets"


        Console.WriteLine("*** BEFORE ***")
        ObjectDumper.Write(q, 1)


        Console.WriteLine()
        Console.WriteLine("*** INSERT ***")
        Dim newCategory = New Category With _
                            {.CategoryName = "Widgets", _
                             .Description = "Widgets are the customer-facing analogues " & _
                                                       "to sprockets and cogs."}

        Dim newProduct = New Product With {.ProductName = "Blue Widget", _
                                           .UnitPrice = New Decimal?(34.56D), _
                                           .Category = newCategory}
        db2.Categories.InsertOnSubmit(newCategory)
        db2.SubmitChanges()


        Console.WriteLine()
        Console.WriteLine("*** AFTER ***")
        ObjectDumper.Write(q, 1)

        Cleanup65()  '还原以前的数据库状态
    End Sub

    Private Sub Cleanup65()
        setLogging(False)

        db.Products.DeleteAllOnSubmit(From prod In db.Products _
                                      Where prod.Category.CategoryName = "Widgets")

        db.Categories.DeleteAllOnSubmit(From category In db.Categories _
                                        Where category.CategoryName = "Widgets")

        db.SubmitChanges()
    End Sub

    <Category("INSERT/UPDATE/DELETE")> _
    <Title("Insert - 多对多")> _
    <Description("此示例使用 Add 方法将新 Employee 添加到 " & _
                 "Employees 表对象，将新 Territory 添加到 Territories 表对象，" & _
                 "并将新 EmployeeTerritory 添加到与此新 Employee 和新 Territory 有外键关系的 " & _
                 "EmployeeTerritories 表对象。" & _
                 "调用 SubmitChanges 将这些新对象及其" & _
                 "关系保持到数据库。")> _
    Public Sub LinqToSqlInsert03()

        Dim db2 = New NorthwindDataContext(My.Settings.NORTHWINDConnectionString1)

        Dim ds = New DataLoadOptions()
        ds.LoadWith(Of Employee)(Function(p) p.EmployeeTerritories)
        ds.LoadWith(Of EmployeeTerritory)(Function(p) p.Territory)

        db2.LoadOptions = ds
        Dim empQuery = From emp In db.Employees _
                       Where emp.FirstName = "Nancy"

        Console.WriteLine("*** BEFORE ***")
        ObjectDumper.Write(empQuery, 1)


        Console.WriteLine()
        Console.WriteLine("*** INSERT ***")
        Dim newEmployee = New Employee With {.FirstName = "Kira", .LastName = "Smith"}
        Dim newTerritory = New Territory With {.TerritoryID = "12345", _
                                               .TerritoryDescription = "Anytown", _
                                               .Region = db.Regions.First()}

        Dim newEmployeeTerritory = New EmployeeTerritory With _
                                    {.Employee = newEmployee, .Territory = newTerritory}
        db.Employees.InsertOnSubmit(newEmployee)
        db.Territories.InsertOnSubmit(newTerritory)
        db.EmployeeTerritories.InsertOnSubmit(newEmployeeTerritory)
        db.SubmitChanges()

        Console.WriteLine()
        Console.WriteLine("*** AFTER ***")
        ObjectDumper.Write(empQuery, 2)

        Cleanup166()  ' 还原以前的数据库状态
    End Sub

    Private Sub Cleanup166()
        setLogging(False)

        db.EmployeeTerritories.DeleteAllOnSubmit(From et In db.EmployeeTerritories _
                                                 Where et.TerritoryID = "12345")
        db.Employees.DeleteAllOnSubmit(From emp In db.Employees _
                                       Where emp.FirstName = "Kira" AndAlso emp.LastName = "Smith")

        db.Territories.DeleteAllOnSubmit(From terr In db.Territories _
                                         Where terr.TerritoryID = "12345")
        db.SubmitChanges()
    End Sub


    <Category("INSERT/UPDATE/DELETE")> _
    <Title("Update - 简单用法")> _
    <Description("此示例使用 SubmitChanges 将对检索到的一个" & _
                 "Customer 对象做出的更新保持回数据库。")> _
    Public Sub LinqToSqlInsert04()
        Dim custQuery = From cust In db.Customers _
                        Where cust.CustomerID = "ALFKI"

        Console.WriteLine("*** BEFORE ***")
        ObjectDumper.Write(custQuery)

        Console.WriteLine()
        Console.WriteLine("*** UPDATE ***")

        Dim custToUpdate = (From cust In db.Customers _
                            Where cust.CustomerID = "ALFKI").First()
        custToUpdate.ContactTitle = "Vice President"

        db.SubmitChanges()

        Console.WriteLine()
        Console.WriteLine("*** AFTER ***")
        ObjectDumper.Write(custQuery)

        Cleanup59()  ' 还原以前的数据库状态
    End Sub

    Private Sub Cleanup59()
        setLogging(False)

        Dim custToUpdate = (From cust In db.Customers _
                            Where cust.CustomerID = "ALFKI").First()

        custToUpdate.ContactTitle = "Sales Representative"
        db.SubmitChanges()
    End Sub

    <Category("INSERT/UPDATE/DELETE")> _
    <Title("Update - 进行多个更改并显示")> _
    <Description("此示例使用 SubmitChanges 将对检索到的" & _
             "Product 对象所做的更新保持回到数据库中。此外，演示如何确定" & _
             "所更改对象的数量、所更改的对象以及所更改的对象成员。")> _
    Public Sub LinqToSqlInsert05()

        Dim query = From p In db.Products _
                    Where p.CategoryID = 1

        Console.WriteLine("*** BEFORE ***")
        ObjectDumper.Write(query)


        Console.WriteLine()
        Console.WriteLine("*** UPDATE ***")
        For Each p In query
            p.UnitPrice += 1D
        Next

        Dim cs As ChangeSet = db.GetChangeSet()

        Console.WriteLine("*** CHANGE SET ***")
        Console.WriteLine("已插入实体的数量: {0}", cs.Inserts.Count)
        Console.WriteLine("Number of entities updated:  {0}", cs.Updates.Count)
        Console.WriteLine("Number of entities deleted:  {0}", cs.Deletes.Count)
        Console.WriteLine()

        Console.WriteLine("*** GetOriginalEntityState ***")
        For Each o In cs.Updates
            Dim p = TryCast(o, Product)

            If p IsNot Nothing Then
                Dim oldP As Product = db.Products.GetOriginalEntityState(p)

                Console.WriteLine("** Current **")
                ObjectDumper.Write(p)

                Console.WriteLine("** Original **")
                ObjectDumper.Write(oldP)

                Console.WriteLine()
            End If
        Next

        Console.WriteLine()

        Console.WriteLine("*** GetModifiedMembers ***")
        For Each o In cs.Updates

            Dim p = TryCast(o, Product)
            If p IsNot Nothing Then

                For Each mmi In db.Products.GetModifiedMembers(p)
                    Console.WriteLine("Member {0}", mmi.Member.Name)
                    Console.WriteLine(vbTab & "Original value: {0}", mmi.OriginalValue)
                    Console.WriteLine(vbTab & "\tCurrent  value: {0}", mmi.CurrentValue)
                Next

            End If
        Next

        db.SubmitChanges()

        Console.WriteLine()
        Console.WriteLine("*** AFTER ***")
        ObjectDumper.Write(query)

        Cleanup68()  '' 还原先前数据库状态
    End Sub


    Private Sub Cleanup68()
        setLogging(False)

        Dim query = From p In db.Products _
                    Where p.CategoryID = 1

        For Each p In query
            p.UnitPrice -= 1D
        Next
        db.SubmitChanges()
    End Sub

    <Category("INSERT/UPDATE/DELETE")> _
    <Title("删除 - 简单用法")> _
    <Description("此示例使用 Remove 方法从 OrderDetails 表对象 " & _
                 "删除 OrderDetail。调用 SubmitChanges 将此" & _
                 "删除保持到数据库。")> _
    Public Sub LinqToSqlInsert06()
        Console.WriteLine("*** BEFORE ***")
        ObjectDumper.Write(From cust In db.Order_Details _
                           Where cust.OrderID = 10255)

        Console.WriteLine()
        Console.WriteLine("*** DELETE ***")
        Dim order = (From cust In db.Order_Details _
                     Where cust.OrderID = 10255 AndAlso cust.ProductID = 36).First()

        db.Order_Details.DeleteOnSubmit(order)
        db.SubmitChanges()


        Console.WriteLine()
        Console.WriteLine("*** AFTER ***")
        clearDBCache()
        ObjectDumper.Write(From cust In db.Order_Details _
                           Where cust.OrderID = 10255 _
                           Select cust)

        Cleanup61()  ' 还原以前的数据库状态
    End Sub

    Private Sub Cleanup61()
        setLogging(False)

        Dim order = New Order_Detail With {.OrderID = 10255, .ProductID = 36, _
                                           .UnitPrice = 15.2D, .Quantity = 25, _
                                           .Discount = 0.0F}

        db.Order_Details.InsertOnSubmit(order)

        db.SubmitChanges()
    End Sub


    <Category("INSERT/UPDATE/DELETE")> _
    <Title("删除 - 一对多")> _
    <Description("此示例使用 Remove 方法从 Order Details 和 Orders 表中删除" & _
                 "Order 和 Order Detail。首先从 Order Details 删除，然后" & _
                 "从 Orders 删除。调用 SubmitChanges 将此删除保持到数据库。")> _
    Public Sub LinqToSqlInsert07()

        Dim orderDetails = From ordDetail In db.Order_Details _
                           Where ordDetail.Order.CustomerID = "WARTH" _
                                 AndAlso ordDetail.Order.EmployeeID = 3

        Console.WriteLine("*** BEFORE ***")
        ObjectDumper.Write(orderDetails)

        Console.WriteLine()
        Console.WriteLine("*** DELETE ***")
        Dim order = (From ord In db.Orders _
                     Where ord.CustomerID = "WARTH" AndAlso ord.EmployeeID = 3).First()

        For Each od As Order_Detail In orderDetails
            db.Order_Details.DeleteOnSubmit(od)
        Next

        db.Orders.DeleteOnSubmit(order)
        db.SubmitChanges()

        Console.WriteLine()
        Console.WriteLine("*** AFTER ***")
        ObjectDumper.Write(orderDetails)

        Cleanup62()  ' 还原以前的数据库状态

    End Sub

    Private Sub Cleanup62()
        setLogging(False)

        Dim order = New Order With { _
                                    .CustomerID = "WARTH", _
                                    .EmployeeID = 3, _
                                    .OrderDate = New DateTime(1996, 7, 26), _
                                    .RequiredDate = New DateTime(1996, 9, 6), _
                                    .ShippedDate = New DateTime(1996, 7, 31), _
                                    .ShipVia = 3, _
                                    .Freight = 25.73D, _
                                    .ShipName = "Wartian Herkku", _
                                    .ShipAddress = "Torikatu 38", _
                                    .ShipCity = "Oulu", _
                                    .ShipPostalCode = "90110", _
                                    .ShipCountry = "Finland"}

        Dim orderDetail = New Order_Detail With { _
                                                 .ProductID = 12, _
                                                 .UnitPrice = 30.4D, _
                                                 .Quantity = 12, _
                                                 .Discount = 0.0F}

        order.Order_Details.Add(orderDetail)
        db.Orders.InsertOnSubmit(order)
        db.SubmitChanges()
    End Sub

    <Category("INSERT/UPDATE/DELETE")> _
    <Title("Insert - 使用 Dynamic CUD 的重写")> _
    <Description("此示例使用 DataContext 提供的分部方法 InsertRegion 来插入一个区域。" & _
                 "对 SubmitChanges 的调用将调用 InsertRegion 重写，该重写使用 Dynamic CUD 来运行 " & _
                 "默认的 Linq To SQL 生成的 SQL 查询")> _
    Public Sub LinqToSqlInsert09()

        Console.WriteLine("*** BEFORE ***")
        ObjectDumper.Write(From c In db.Regions _
                           Where c.RegionID = 32)

        Console.WriteLine()
        Console.WriteLine("*** INSERT OVERRIDE ***")

        ''饮料
        Dim nwRegion As New Region() With {.RegionID = 32, .RegionDescription = "Rainy"}

        db.Regions.InsertOnSubmit(nwRegion)
        db.SubmitChanges()


        Console.WriteLine()
        Console.WriteLine("*** AFTER ***")
        clearDBCache()

        ObjectDumper.Write(From c In db.Regions _
                           Where c.RegionID = 32)

        CleanupInsert09() '' 还原先前数据库状态
    End Sub

    Private Sub CleanupInsert09()


        setLogging(False)

        db.Regions.DeleteAllOnSubmit(From r In db.Regions _
                                     Where r.RegionID = 32)
        db.SubmitChanges()
    End Sub

    <Category("INSERT/UPDATE/DELETE")> _
    <Title("先使用 Attach 再使用 Update")> _
    <Description("此示例从一个上下文中获取实体，使用 Attach 和 AttachAll " & _
                 "附加来自另一个上下文的实体，然后更新实体。 " & _
                 "更改被提交到数据库。")> _
    Public Sub LinqToSqlInsert10()

        '' 通常通过从另一个层的反序列化 XML 中获取要附加的实体。
        Dim c1 As Customer
        Dim c2Orders As List(Of Order)

        Using tempdb As New NorthwindDataContext(My.Settings.NORTHWINDConnectionString1)

            c1 = tempdb.Customers.Single(Function(c) c.CustomerID = "ALFKI")
            Console.WriteLine("Customer {0}'s original address {1}", c1.CustomerID, c1.Address)
            Console.WriteLine()

            Dim tempcust = tempdb.Customers.Single(Function(c) c.CustomerID = "ANTON")
            c2Orders = tempcust.Orders.ToList()
            For Each o In c2Orders
                Console.WriteLine("Order {0} belongs to customer {1}", o.OrderID, o.CustomerID)
            Next

            Console.WriteLine()

            Dim tempcust2 = tempdb.Customers.Single(Function(c) c.CustomerID = "CHOPS")
            Dim c3Orders = tempcust2.Orders.ToList()
            For Each o In c3Orders
                Console.WriteLine("Order {0} belongs to customer {1}", o.OrderID, o.CustomerID)
            Next

            Console.WriteLine()
        End Using

        Using db2 As New NorthwindDataContext(My.Settings.NORTHWINDConnectionString1)

            '' 将第一个实体附加到当前数据上下文以跟踪更改。
            db2.Customers.Attach(c1)
            Console.WriteLine("***** Update Customer ALFKI's address ******")
            Console.WriteLine()

            '' 更改被跟踪的实体。
            c1.Address = "123 First Ave"

            '' 附加定单列表中的所有实体。
            db2.Orders.AttachAll(c2Orders)

            '' 更新属于另一个客户的定单。
            Console.WriteLine("****** Assign all Orders belong to ANTON to CHOPS ******")
            Console.WriteLine()

            For Each o In c2Orders
                o.CustomerID = "CHOPS"
            Next


            '' 提交当前数据上下文中的更改。
            db2.SubmitChanges()
        End Using

        '' 检查是否按照预期提交了定单。
        Using db3 As New NorthwindDataContext(My.Settings.NORTHWINDConnectionString1)

            Dim dbC1 = db3.Customers.Single(Function(c) c.CustomerID = "ALFKI")

            Console.WriteLine("Customer {0}'s new address {1}", dbC1.CustomerID, dbC1.Address)
            Console.WriteLine()

            Dim dbC2 = db3.Customers.Single(Function(c) c.CustomerID = "CHOPS")

            For Each o In dbC2.Orders
                Console.WriteLine("Order {0} belongs to customer {1}", o.OrderID, o.CustomerID)
            Next

        End Using

        CleanupInsert10()
    End Sub


    Private Sub CleanupInsert10()

        Dim c2OrderIDs = New Integer() {10365, 10507, 10535, 10573, 10677, 10682, 10856}

        Using tempdb As New NorthwindDataContext(My.Settings.NORTHWINDConnectionString1)

            Dim c1 = tempdb.Customers.Single(Function(c) c.CustomerID = "ALFKI")
            c1.Address = "Obere Str. 57"

            For Each o In tempdb.Orders.Where(Function(p) c2OrderIDs.Contains(p.OrderID))
                o.CustomerID = "ANTON"
            Next
            tempdb.SubmitChanges()
        End Using

    End Sub

    <Category("INSERT/UPDATE/DELETE")> _
    <Title("先使用 Attach 再使用 Update 和 Delete")> _
    <Description("此示例从一个上下文中获取实体，并使用 Attach 和 AttachAll " & _
                 "来附加来自另一上下文的实体。然后，更新两个实体，" & _
                 "删除了一个实体，并且添加了另一个实体。更改被提交到" & _
                 "数据库")> _
    Public Sub LinqToSqlInsert11()

        '' 通常，通过从另一个层的反序列化 XML 中
        '' 获取要附加的实体。
        '' 此示例使用 LoadWith 来在一个查询中预先加载客户和定单
        '' 并且禁用延迟的加载。
        Dim cust As Customer = Nothing
        Using tempdb As NorthwindDataContext = New NorthwindDataContext(My.Settings.NORTHWINDConnectionString1)

            Dim shape As New DataLoadOptions()
            shape.LoadWith(Of Customer)(Function(c) c.Orders)
            '' 加载第一个客户实体及其定单。
            tempdb.LoadOptions = shape
            tempdb.DeferredLoadingEnabled = False
            cust = tempdb.Customers.First(Function(x) x.CustomerID = "ALFKI")
        End Using

        Console.WriteLine("Customer {0}'s original phone number {1}", cust.CustomerID, cust.Phone)
        Console.WriteLine()

        For Each o In cust.Orders
            Console.WriteLine("Customer {0} has order {1} for city {2}", _
                              o.CustomerID, o.OrderID, o.ShipCity)
        Next

        Dim orderA = cust.Orders.First()
        Dim orderB = cust.Orders.First(Function(x) x.OrderID > orderA.OrderID)

        Using db2 As New NorthwindDataContext(My.Settings.NORTHWINDConnectionString1)

            '' 将第一个实体附加到当前数据上下文以跟踪更改。
            db2.Customers.Attach(cust)
            '' 附加相关定单以便进行跟踪；否则，将在提交时插入这些定单。
            db2.Orders.AttachAll(cust.Orders.ToList())

            '' 更新客户。
            cust.Phone = "2345 5436"
            '' 更新第一个定单。
            orderA.ShipCity = "Redmond"
            '' 删除第二个定单。
            cust.Orders.Remove(orderB)
            '' 创建新定单并为客户添加此定单。
            Dim orderC = New Order() With {.ShipCity = "New York"}
            Console.WriteLine("Adding new order")
            cust.Orders.Add(orderC)

            '' 现在，提交所有更改
            db2.SubmitChanges()
        End Using


        '' 验证是否按照预期应用了更改。
        Using db3 As New NorthwindDataContext(My.Settings.NORTHWINDConnectionString1)

            Dim newCust = db3.Customers.First(Function(x) x.CustomerID = "ALFKI")
            Console.WriteLine("Customer {0}'s new phone number {1}", newCust.CustomerID, newCust.Phone)
            Console.WriteLine()

            For Each o In newCust.Orders
                Console.WriteLine("Customer {0} has order {1} for city {2}", o.CustomerID, o.OrderID, o.ShipCity)
            Next

        End Using


        CleanupInsert11()
    End Sub


    Private Sub CleanupInsert11()
        Dim alfkiOrderIDs = New Integer() {10643, 10692, 10702, 10835, 10952, 11011}

        Using tempdb As New NorthwindDataContext(My.Settings.NORTHWINDConnectionString1)
            Dim c1 = tempdb.Customers.Single(Function(c) c.CustomerID = "ALFKI")
            c1.Phone = "030-0074321"

            Dim oa = tempdb.Orders.Single(Function(o) o.OrderID = 10643)
            oa.ShipCity = "Berlin"

            Dim ob = tempdb.Orders.Single(Function(o) o.OrderID = 10692)
            ob.CustomerID = "ALFKI"

            For Each o In c1.Orders.Where(Function(p) Not alfkiOrderIDs.Contains(p.OrderID))
                tempdb.Orders.DeleteOnSubmit(o)
            Next
            tempdb.SubmitChanges()
        End Using

    End Sub

    <Category("同时更改")> _
    <Title("开放式并发 - 1")> _
    <Description("这个示例和下一个示例演示开放式并发。在此示例中，" & _
                 "其他用户在您读取数据之前对 Product 1 做出更新并将其更新提交，" & _
                 "因此未发生冲突。")> _
    Public Sub LinqToSqlSimultaneous01()
        Console.WriteLine("OTHER USER: ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~")

        ' 打开与数据库的第二个连接以模拟
        ' 将要对 Products 表进行更改的另一个用户
        Dim otherUser_db As NorthwindDataContext = New NorthwindDataContext()
        otherUser_db.Log = db.Log

        Dim otherUser_product = (From prod In otherUser_db.Products _
                                 Where prod.ProductID = 1).First()

        otherUser_product.UnitPrice = 999.99D
        otherUser_db.SubmitChanges()

        Console.WriteLine("~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~")


        Console.WriteLine()
        Console.WriteLine("YOU:  ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~")

        Dim product = (From prod In db.Products _
                       Where prod.ProductID = 1).First()

        product.UnitPrice = 777.77D

        Dim conflict As Boolean = False
        Try
            db.SubmitChanges()
        Catch x As DBConcurrencyException
            conflict = True
        End Try

        Console.WriteLine()
        If (conflict) Then
            Console.WriteLine("* * * OPTIMISTIC CONCURRENCY EXCEPTION * * *")
            Console.WriteLine("Another user has changed Product 1 since it was first requested.")
            Console.WriteLine("Backing out changes.")
        Else
            Console.WriteLine("* * * COMMIT SUCCESSFUL * * *")
            Console.WriteLine("Changes to Product 1 saved.")
        End If

        Console.WriteLine("~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~")



        Cleanup63()  ' 还原以前的数据库状态
    End Sub

    Private Sub Cleanup63()
        clearDBCache()
        setLogging(False)

        Dim product = (From prod In db.Products _
                       Where prod.ProductID = 1).First()

        product.UnitPrice = 18D
        db.SubmitChanges()
    End Sub

    <Category("同时更改")> _
    <Title("开放式并发 - 2")> _
    <Description("这个示例和上一个示例演示开放式并发。在此示例中，" & _
                 "其他用户在您读取数据之后、完成更新之前对 Product 1 做出更新并将其更新提交，" & _
                 "因此导致了开放式并发冲突。" & _
                 "此时将回滚您的更改，使您可以从数据库检索新更新的数据，" & _
                 "并决定如何继续进行您自己的更新。")> _
    Public Sub LinqToSqlSimultaneous02()
        Console.WriteLine("YOU:  ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~")

        Dim product = (From prod In db.Products _
                       Where prod.ProductID = 1 _
                       Select prod).First()

        Console.WriteLine("~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~")


        Console.WriteLine()
        Console.WriteLine("OTHER USER: ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~")

        ' 打开与数据库的第二个连接以模拟
        ' 将要对 Products 表进行更改的另一个用户
        Dim otherUser_db As NorthwindDataContext = New NorthwindDataContext()
        otherUser_db.Log = db.Log

        Dim otherUser_product = (From prod In otherUser_db.Products _
                                 Where prod.ProductID = 1 _
                                 Select prod).First()
        otherUser_product.UnitPrice = 999.99D
        otherUser_db.SubmitChanges()

        Console.WriteLine("~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~")


        Console.WriteLine()
        Console.WriteLine("YOU (continued): ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~")

        product.UnitPrice = 777.77D

        Dim conflict As Boolean = False
        Try
            db.SubmitChanges()
        Catch x As DBConcurrencyException
            conflict = True
        End Try

        Console.WriteLine()
        If (conflict) Then
            Console.WriteLine("* * * OPTIMISTIC CONCURRENCY EXCEPTION * * *")
            Console.WriteLine("Another user has changed Product 1 since it was first requested.")
            Console.WriteLine("Backing out changes.")
        Else
            Console.WriteLine("* * * COMMIT SUCCESSFUL * * *")
            Console.WriteLine("Changes to Product 1 saved.")
        End If

        Console.WriteLine("~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ")



        Cleanup64()  ' 还原以前的数据库状态
    End Sub

    Private Sub Cleanup64()
        clearDBCache()
        setLogging(False)

        ' 事务失败后将自动回滚数据
    End Sub

    <Category("同时更改")> _
    <Title("事务 - 隐式")> _
    <Description("此示例演示由 SubmitChanges 创建的" & _
                 "隐式事务。更新 prod2 的 UnitsInStock 字段使" & _
                 "该字段为负值，而这违反了" & _
                 "服务器上的 CHECK 约束。这导致" & _
                 "更新这两个产品的事务失败，此时将回滚所有更改。")> _
    Public Sub LinqToSqlSimultaenous03()
        Console.WriteLine("*** BEFORE ***")
        ObjectDumper.Write(From prod In db.Products _
                           Where prod.ProductID = 4)

        ObjectDumper.Write(From prod In db.Products _
                           Where prod.ProductID = 5)

        Console.WriteLine()
        Console.WriteLine("*** UPDATE WITH IMPLICIT TRANSACTION ***")
        Try
            Dim prod1 As Product = (From prod In db.Products _
                                    Where prod.ProductID = 4 _
                                    Select prod).First()
            Dim prod2 As Product = (From prod In db.Products _
                                    Where prod.ProductID = 5 _
                                    Select prod).First()
            prod1.UnitsInStock = prod1.UnitsInStock - 3S
            prod2.UnitsInStock = prod2.UnitsInStock - 5S    ' 错误: 这将使库存量变为负值
            ' db.SubmitChanges 隐式使用事务，以便
            ' 同时接受或同时拒绝两个更新
            db.SubmitChanges()
        Catch e As Exception
            Console.WriteLine(e.Message)
        End Try


        Console.WriteLine()
        Console.WriteLine("*** AFTER ***")
        clearDBCache()
        ObjectDumper.Write(From prod In db.Products _
                           Where prod.ProductID = 4 _
                           Select prod)
        ObjectDumper.Write(From prod In db.Products _
                           Where prod.ProductID = 5 _
                           Select prod)



        Cleanup95()  ' 还原以前的数据库状态
    End Sub

    Private Sub Cleanup95()
        setLogging(False)

        ' 事务失败后将自动回滚数据
    End Sub

    <Category("同时更改")> _
    <Title("事务 - 显式")> _
    <Description("此示例演示使用显式事务。" & _
                 "通过在事务中加入对数据的读取以防止出现开放式并发异常，" & _
                 "显式事务可以提供更多的保护。" & _
                 "如同上一个查询中，更新 prod2 的 UnitsInStock 字段将使" & _
                 "该字段为负值，而这违反了" & _
                 "数据库中的 CHECK 约束。这导致" & _
                 "更新这两个产品的事务失败，此时将回滚所有更改。")> _
    Public Sub LinqToSqlSimultaneous04()
        Console.WriteLine("*** BEFORE ***")
        ObjectDumper.Write(From prod In db.Products _
                           Where prod.ProductID = 4 _
                           Select prod)
        ObjectDumper.Write(From prod In db.Products _
                           Where prod.ProductID = 5 _
                           Select prod)


        Console.WriteLine()
        Console.WriteLine("*** UPDATE WITH EXPLICIT TRANSACTION ***")
        ' 显式使用 TransactionScope 可以确保
        ' 在读写之间数据库中的数据
        ' 不发生变化

        Using ts As TransactionScope = New TransactionScope()
            Try
                Dim prod1 As Product = (From p In db.Products _
                                        Where p.ProductID = 4 _
                                        Select p).First()
                Dim prod2 As Product = (From prod In db.Products _
                                        Where prod.ProductID = 5 _
                                        Select prod).First()
                prod1.UnitsInStock = prod1.UnitsInStock - 3S
                prod2.UnitsInStock = prod2.UnitsInStock - 5S    ' 错误: 这将使库存量变为负值
                db.SubmitChanges()
            Catch e As Exception
                Console.WriteLine(e.Message)
            End Try
        End Using

        Console.WriteLine()
        Console.WriteLine("*** AFTER ***")
        clearDBCache()
        ObjectDumper.Write(From prod In db.Products _
                           Where prod.ProductID = 4 _
                           Select prod)
        ObjectDumper.Write(From prod In db.Products _
                           Where prod.ProductID = 5 _
                           Select prod)

        Cleanup66()  ' 还原以前的数据库状态
    End Sub

    Private Sub Cleanup66()
        setLogging(False)

        ' 事务失败后将自动回滚数据
    End Sub

    <Category("NULL")> _
    <Title("处理 NULL (在 VB 中为 Nothing)")> _
    <Description("此示例使用 Nothing 值查找" & _
                 "不隶属于另一个雇员的所有雇员。")> _
    Public Sub LinqToSqlNull01()
        Dim empQuery = From emp In db.Employees _
                       Where emp.ReportsTo Is Nothing

        ObjectDumper.Write(empQuery)
    End Sub

    <Category("NULL")> _
    <Title("Nullable(Of T).HasValue")> _
    <Description("此示例使用 Nullable(Of T).HasValue 查找" & _
                 "不隶属于另一个雇员的所有雇员。")> _
    Public Sub LinqToSqlNull02()
        Dim empQuery = From emp In db.Employees _
                       Where Not emp.ReportsTo.HasValue _
                       Select emp

        ObjectDumper.Write(empQuery)
    End Sub

    <Category("NULL")> _
    <Title("Nullable(Of T).Value")> _
    <Description("此示例对隶属于另一个雇员的雇员使用 Nullable(Of T).Value" & _
                 "返回前者的 EmployeeID 编号。" & _
                 "请注意，" & _
                 ".Value 为可选。")> _
    Public Sub LinqToSqlNull03()
        Dim empQuery = From emp In db.Employees _
                       Where emp.ReportsTo.HasValue _
                       Select emp.FirstName, emp.LastName, ReportsTo = emp.ReportsTo.Value

        ObjectDumper.Write(empQuery)
    End Sub

    <Category("字符串/日期函数")> _
    <Title("字符串串联")> _
    <Description("此示例使用 & 运算符在形成" & _
                 "经计算得出的客户 Location 值过程中将字符串字段和" & _
                 "字符串串联在一起。")> _
    Public Sub LinqToSqlString01()
        Dim custQuery = From cust In db.Customers _
                        Select cust.CustomerID, _
                               Location = cust.City & ", " & cust.Country

        ObjectDumper.Write(custQuery, 1)
    End Sub

    <Category("字符串/日期函数")> _
    <Title("String.Length")> _
    <Description("此示例使用 Length 属性查找名称短于 " & _
                  "10 个字符的所有产品。")> _
     Public Sub LinqToSqlString02()
        Dim shortProducts = From prod In db.Products _
                            Where prod.ProductName.Length < 10

        ObjectDumper.Write(shortProducts)
    End Sub

    <Category("字符串/日期函数")> _
    <Title("String.Contains(substring)")> _
    <Description("此示例使用 Contains 方法查找所有其联系人姓名中" & _
                 "包含“Anders”的客户。")> _
    Public Sub LinqToSqlString03()
        Dim custQuery = From cust In db.Customers _
                        Where cust.ContactName.Contains("Anders")

        ObjectDumper.Write(custQuery)
    End Sub

    <Category("字符串/日期函数")> _
    <Title("String.IndexOf(substring)")> _
    <Description("此示例使用 IndexOf 方法查找每个客户" & _
                 "联系人姓名中出现第一个空格的位置。")> _
    Public Sub LinqToSqlString04()
        Dim custQuery = From cust In db.Customers _
                        Select cust.ContactName, SpacePos = cust.ContactName.IndexOf(" ")

        ObjectDumper.Write(custQuery)
    End Sub

    <Category("字符串/日期函数")> _
    <Title("String.StartsWith(prefix)")> _
    <Description("此示例使用 StartsWith 方法查找" & _
                 "联系人姓名以“Maria”开头的客户。")> _
    Public Sub LinqToSqlString05()
        Dim custQuery = From cust In db.Customers _
                        Where cust.ContactName.StartsWith("Maria")

        ObjectDumper.Write(custQuery)
    End Sub

    <Category("字符串/日期函数")> _
    <Title("String.EndsWith(suffix)")> _
    <Description("此示例使用 StartsWith 方法查找" & _
                 "联系人姓名以“Anders”结尾的客户。")> _
    Public Sub LinqToSqlString06()
        Dim custQuery = From cust In db.Customers _
                        Where cust.ContactName.EndsWith("Anders")

        ObjectDumper.Write(custQuery)
    End Sub

    <Category("字符串/日期函数")> _
    <Title("String.Substring(start)")> _
    <Description("此示例使用 Substring 方法返回产品名称中" & _
                 "从第四个字母开始的部分。")> _
    Public Sub LinqToSqlString07()
        Dim prodQuery = From prod In db.Products _
                        Select prod.ProductName.Substring(3)

        ObjectDumper.Write(prodQuery)
    End Sub

    <Category("字符串/日期函数")> _
    <Title("String.Substring(start, length)")> _
    <Description("此示例使用 Substring 方法查找" & _
                 "家庭电话号码第七位到第九位是“555”的雇员。")> _
    Public Sub LinqToSqlString08()
        Dim empQuery = From emp In db.Employees _
                       Where emp.HomePhone.Substring(6, 3) = "555"

        ObjectDumper.Write(empQuery)
    End Sub

    <Category("字符串/日期函数")> _
    <Title("String.ToUpper()")> _
    <Description("此示例使用 ToUpper 方法返回" & _
                 "姓氏已转换为大写的雇员姓名。")> _
    Public Sub LinqToSqlString09()
        Dim empQuery = From emp In db.Employees _
                       Select LastName = emp.LastName.ToUpper(), emp.FirstName

        ObjectDumper.Write(empQuery)
    End Sub

    <Category("字符串/日期函数")> _
    <Title("String.ToLower()")> _
    <Description("此示例使用 ToLower 方法返回" & _
                 "已转换为小写的类别名称。")> _
    Public Sub LinqToSqlString10()
        Dim categoryQuery = From category In db.Categories _
                            Select category.CategoryName.ToLower()

        ObjectDumper.Write(categoryQuery)
    End Sub

    <Category("字符串/日期函数")> _
    <Title("String.Trim()")> _
    <Description("此示例使用 Trim 方法返回雇员家庭电话号码的" & _
                 "前五位，并移除前导和" & _
                 "尾随空格。")> _
    Public Sub LinqToSqlString11()
        Dim empQuery = From emp In db.Employees _
                       Select emp.HomePhone.Substring(0, 5).Trim()

        ObjectDumper.Write(empQuery)
    End Sub

    <Category("字符串/日期函数")> _
    <Title("String.Insert(pos, str)")> _
    <Description("此示例使用 Insert 方法返回第五位" & _
                 "为 ) 的雇员电话号码的序列，" & _
                 "并在 ) 后面插入一个 :。")> _
    Public Sub LinqToSqlString12()
        Dim empQuery = From emp In db.Employees _
                       Where emp.HomePhone.Substring(4, 1) = ")" _
                       Select emp.HomePhone.Insert(5, ":")

        ObjectDumper.Write(empQuery)
    End Sub

    <Category("字符串/日期函数")> _
    <Title("String.Remove(start)")> _
    <Description("此示例使用 Insert 方法返回第五位" & _
                 "为 ) 的雇员电话号码的序列，" & _
                 "并移除从第十个字符开始的所有字符。")> _
    Public Sub LinqToSqlString13()
        Dim empQuery = From emp In db.Employees _
                       Where emp.HomePhone.Substring(4, 1) = ")" _
                       Select emp.HomePhone.Remove(9)

        ObjectDumper.Write(empQuery)
    End Sub

    <Category("字符串/日期函数")> _
    <Title("String.Remove(start, length)")> _
    <Description("此示例使用 Insert 方法返回第五位" & _
                 "为 ) 的雇员电话号码的序列，" & _
                 "并移除前六个字符。")> _
    Public Sub LinqToSqlString14()
        Dim empQuery = From emp In db.Employees _
                       Where emp.HomePhone.Substring(4, 1) = ")" _
                       Select emp.HomePhone.Remove(0, 6)

        ObjectDumper.Write(empQuery)
    End Sub

    <Category("字符串/日期函数")> _
    <Title("String.Replace(find, replace)")> _
    <Description("此示例使用 Replace 方法返回 Country 字段中" & _
                 "UK 被替换为 United Kingdom 以及" & _
                 "USA 被替换为 United States of America 的" & _
                 "供应商信息的序列。")> _
    Public Sub LinqToSqlString15()
        Dim supplierQuery = From supplier In db.Suppliers _
                            Select supplier.CompanyName, _
                                   Country = supplier.Country.Replace("UK", "United Kingdom") _
                                                             .Replace("USA", "United States of America")

        ObjectDumper.Write(supplierQuery)
    End Sub

    <Category("字符串/日期函数")> _
    <Title("DateTime.Year")> _
    <Description("此示例使用 DateTime 的 Year 属性" & _
                 "查找 1997 年下的订单。")> _
    Public Sub LinqToSqlString16()
        Dim ordersIn97 = From ord In db.Orders _
                         Where ord.OrderDate.Value.Year = 1997

        ObjectDumper.Write(ordersIn97)
    End Sub

    <Category("字符串/日期函数")> _
    <Title("DateTime.Month")> _
    <Description("此示例使用 DateTime 的 Month 属性" & _
                 "查找十二月下的订单。")> _
    Public Sub LinqToSqlString17()
        Dim decemberOrders = From ord In db.Orders _
                             Where ord.OrderDate.Value.Month = 12

        ObjectDumper.Write(decemberOrders)
    End Sub

    <Category("字符串/日期函数")> _
    <Title("DateTime.Day")> _
    <Description("此示例使用 DateTime 的 Day 属性" & _
                 "查找某月 31 日下的订单。")> _
    Public Sub LinqToSqlString18()
        Dim ordQuery = From ord In db.Orders _
                       Where ord.OrderDate.Value.Day = 31

        ObjectDumper.Write(ordQuery)
    End Sub

    <Category("对象标识")> _
    <Title("对象缓存 - 1")> _
    <Description("此示例演示在两次执行相同查询时，是如何" & _
                 "每次都引用内存中相同对象的。")> _
    Public Sub LinqToSqlObjectIdentity01()
        Dim cust1 = db.Customers.First(Function(cust) cust.CustomerID = "BONAP")
        Dim cust2 = (From cust In db.Customers _
                                 Where cust.CustomerID = "BONAP").First()

        Console.WriteLine("cust1 and cust2 refer to the same object in memory: " & _
                          Object.ReferenceEquals(cust1, cust2))
    End Sub

    <Category("对象标识")> _
    <Title("对象缓存 - 2")> _
    <Description("此示例演示在执行返回" & _
                 "数据库中相同行的不同查询时，" & _
                 "是如何每次都引用内存中相同对象的。")> _
    Public Sub LinqToSqlObjectIdentity02()
        Dim cust1 = db.Customers.First(Function(cust) cust.CustomerID = "BONAP")
        Dim cust2 = (From ord In db.Orders _
                     Where ord.Customer.CustomerID = "BONAP").First().Customer

        Console.WriteLine("cust1 and cust2 refer to the same object in memory: " & _
                          Object.ReferenceEquals(cust1, cust2))
    End Sub

    <Category("对象加载")> _
    <Title("延迟加载 - 1")> _
    <Description("此示例演示如果原始查询未请求数据，" & _
                 "在所检索到对象的各个关系中导航" & _
                 "如何能停止触发对数据库的新查询。")> _
    Public Sub LinqToSqlObject01()
        Dim custs = From cust In db.Customers _
                    Where cust.City = "Sao Paulo" _
                    Select cust

        For Each cust In custs
            For Each ord In cust.Orders
                Console.WriteLine("CustomerID " & cust.CustomerID & " has an OrderID " & ord.OrderID)
            Next
        Next
    End Sub

    <Category("对象加载")> _
    <Title("LoadWith - 预先加载 - 1")> _
    <Description("此示例演示如何在原始查询过程中" & _
                 "使用 LoadWith 请求相关数据，" & _
                 "以便稍后在检索到的各个对象中导航时" & _
                 "不需要对数据库进行额外的往返。")> _
    Public Sub LinqToSqlObject02()

        Dim db2 = New NorthwindDataContext(My.Settings.NORTHWINDConnectionString1)
        db2.Log = Me.OutputStreamWriter

        Dim ds = New DataLoadOptions()
        ds.LoadWith(Of Customer)(Function(cust) cust.Orders)

        db2.LoadOptions = ds

        Dim custs = From cust In db.Customers _
                    Where cust.City = "Sao Paulo"

        For Each cust In custs
            For Each ord In cust.Orders
                Console.WriteLine("CustomerID " & cust.CustomerID & " has an OrderID " & ord.OrderID)
            Next
        Next

    End Sub

    <Category("对象加载")> _
    <Title("延迟加载 + AssociateWith")> _
    <Description("此示例演示如果原始查询未请求数据，" & _
                 "在所检索到对象的各个关系中导航" & _
                 "如果原始查询没有请求该数据。此外，此示例还演示在延迟了关系对象的加载时，" & _
                 "可以使用 AssoicateWith 筛选这些对象。")> _
    Public Sub LinqToSqlObject03()

        Dim db2 As New NorthwindDataContext(My.Settings.NORTHWINDConnectionString1)
        db2.Log = Me.OutputStreamWriter

        Dim ds As New DataLoadOptions()
        ds.AssociateWith(Of Customer)(Function(p) p.Orders.Where(Function(o) o.ShipVia.Value > 1))

        db2.LoadOptions = ds

        Dim custs = From cust In db2.Customers _
                    Where cust.City = "London"

        For Each cust In custs
            For Each ord In cust.Orders
                For Each orderDetail In ord.Order_Details
                    Console.WriteLine("CustomerID {0} has an OrderID {1} that ShipVia is {2} with ProductID {3} that has name {4}.", _
                        cust.CustomerID, ord.OrderID, ord.ShipVia, orderDetail.ProductID, orderDetail.Product.ProductName)
                Next
            Next
        Next
    End Sub

    <Category("对象加载")> _
    <Title("LoadWith - 预先加载 + AssociateWith")> _
    <Description("此示例演示如何在原始查询过程中" & _
                 "使用 LoadWith 请求相关数据，" & _
                 "以便稍后在检索到的各个对象中导航时" & _
                 "检索的对象。此外，此示例还演示" & _
                 "在对关系对象进行预先加载时，可以使用 Assoicate With 排序关系对象。")> _
    Public Sub LinqToSqlObject04()

        Dim db2 = New NorthwindDataContext(My.Settings.NORTHWINDConnectionString1)
        db2.Log = Me.OutputStreamWriter

            
        Dim ds As New DataLoadOptions()
        ds.LoadWith(Of Customer)(Function(cust) cust.Orders)
        ds.LoadWith(Of Order)(Function(ord) ord.Order_Details)

        ds.AssociateWith(Of Order)(Function(p) p.Order_Details.OrderBy(Function(o) o.Quantity))

        db2.LoadOptions = ds

        Dim custs = From cust In db.Customers _
                    Where cust.City = "London"

        For Each cust In custs
            For Each ord In cust.Orders
                For Each orderDetail In ord.Order_Details
                    Console.WriteLine("CustomerID {0} has an OrderID {1} with ProductID {2} that has quantity {3}.", _
                        cust.CustomerID, ord.OrderID, orderDetail.ProductID, orderDetail.Quantity)
                Next
            Next
        Next

    End Sub


    Private Function isValidProduct(ByVal prod As Product) As Boolean
        Return (prod.ProductName.LastIndexOf("C") = 0)
    End Function

    <Category("对象加载")> _
    <Title("延迟加载 - (1:M)")> _
    <Description("此示例演示如果原始查询未请求数据，" & _
                 "在所检索到对象的各个关系中导航" & _
                 "如何能导致触发对数据库的新查询。")> _
    Public Sub LinqToSqlObject05()
        Dim emps = db.Employees

        For Each emp In emps
            For Each man In emp.Employees
                Console.WriteLine("Employee " & emp.FirstName & " reported to Manager " & man.FirstName)
            Next
        Next
    End Sub

    <Category("对象加载")> _
    <Title("延迟加载 - (BLOB)")> _
    <Description("此示例演示如果数据类型为 Link，" & _
                 "在所检索到各个对象的链接中导航" & _
                 "如何能停止触发对数据库的新查询。")> _
    Public Sub LinqToSqlObject06()
        Dim emps = db.Employees

        For Each emp In emps
            Console.WriteLine(emp.Notes)
        Next

    End Sub


    <Category("对象加载")> _
    <Title("加载重写")> _
    <Description("此示例重写 Category 类的分部方法  LoadProducts。在要加载某个类别的产品时，" & _
                 "将调用 LoadProducts 加载该类别中不连续的产品。")> _
    Public Sub LinqToSqlObject07()

        Dim db2 As New NorthwindDataContext(My.Settings.NORTHWINDConnectionString1)

        Dim ds As New DataLoadOptions()

        ds.LoadWith(Of Category)(Function(p) p.Products)
        db2.LoadOptions = ds

        Dim q = From c In db2.Categories _
                Where c.CategoryID < 3

        For Each cat In q
            For Each prod In cat.Products
                Console.WriteLine("Category {0} has a ProductID {1} that Discontined = {2}.", _
                                  cat.CategoryID, prod.ProductID, prod.Discontinued)
            Next
        Next

    End Sub


    <Category("转换运算符")> _
    <Title("AsEnumerable")> _
    <Description("此示例使用 ToArray，以便使用 Where 的" & _
                 "客户端 IEnumerable(Of T) 实现，而非" & _
                 "将在服务器上转换为 SQL 并执行的" & _
                 "默认 Query(Of T) 实现。这很有必要，因为 Where 子句" & _
                 "引用了用户定义的客户端方法 isValidProduct，" & _
                 "该方法不能转换为 SQL。")> _
    <LinkedFunction("isValidProduct")> _
    Public Sub LinqToSqlConversions01()
        Dim prodQuery = From prod In db.Products.AsEnumerable() _
                        Where isValidProduct(prod)

        ObjectDumper.Write(prodQuery)
    End Sub

    <Category("转换运算符")> _
    <Title("ToArray")> _
    <Description("此示例使用 ToArray 将查询直接计算为数组，" & _
                 "并获取第 3 个元素。")> _
    Public Sub LinqToSqlConversions02()
        Dim londonCustomers = From cust In db.Customers _
                              Where cust.City = "London"

        Dim custArray = londonCustomers.ToArray()
        ObjectDumper.Write(custArray(3), 0)
    End Sub

    <Category("转换运算符")> _
    <Title("ToList")> _
    <Description("此示例使用 ToList 将查询直接计算为 List(Of T)。")> _
    Public Sub LinqToSqlConversions03()
        Dim hiredAfter1994 = From emp In db.Employees _
                             Where emp.HireDate >= #1/1/1994#

        Dim qList = hiredAfter1994.ToList()
        ObjectDumper.Write(qList, 0)
    End Sub

    <Category("转换运算符")> _
    <Title("ToDictionary")> _
    <Description("此示例使用 ToDictionary 将查询和键表达式直接" & _
                 "计算为 Dictionary(Of K, T)。")> _
    Public Sub LinqToSqlConversion04()
        Dim prodQuery = From prod In db.Products _
                        Where prod.UnitsInStock <= prod.ReorderLevel _
                              AndAlso Not prod.Discontinued

        Dim qDictionary = prodQuery.ToDictionary(Function(prod) prod.ProductID)

        For Each key In qDictionary.Keys
            Console.WriteLine("Key " & key & ":")
            ObjectDumper.Write(qDictionary(key))
            Console.WriteLine()
        Next
    End Sub


    <Category("直接 SQL")> _
    <Title("SQL 查询")> _
    <Description("此示例使用 ExecuteQuery(Of T) 执行任意 SQL 查询，" & _
                 "并将所得的行映射为 Product 对象的序列。")> _
    Public Sub LinqToSqlDirect01()
        Dim products = db.ExecuteQuery(Of Product)( _
            "SELECT [Product List].ProductID, [Product List].ProductName " & _
            "FROM Products AS [Product List] " & _
            "WHERE [Product List].Discontinued = 0 " & _
            "ORDER BY [Product List].ProductName;")

        ObjectDumper.Write(products)
    End Sub

    <Category("直接 SQL")> _
    <Title("SQL 命令")> _
    <Description("此示例使用 ExecuteCommand 执行任意 SQL 命令，" & _
                 "本例中为将所有产品单价提高 1.00 的批量更新。")> _
    Public Sub LinqToSqlDirect02()
        Console.WriteLine("*** BEFORE ***")
        ObjectDumper.Write(From prod In db.Products _
                           Select prod)


        Console.WriteLine()
        Console.WriteLine("*** UPDATE ***")
        db.ExecuteQuery(Of Object)("UPDATE Products SET UnitPrice = UnitPrice + 1.00")

        Console.WriteLine()
        Console.WriteLine("*** AFTER ***")
        clearDBCache()
        ObjectDumper.Write(From prod In db.Products _
                           Select prod)

        Cleanup99()  ' 还原以前的数据库状态
    End Sub

    Private Sub Cleanup99()
        setLogging(False)
    End Sub

    <Category("ADO.NET 互操作")> _
    <Title("连接互操作")> _
    <Description("此示例使用预先存在的 ADO.NET 连接创建可用于执行查询的 Northwind 对象，" & _
                 "本例中的查询返回" & _
                 "运费至少为 500.00 的所有订单。")> _
    Public Sub LinqToSqlAdo01()
        ' 创建标准 ADO.NET 连接:
        Dim nwindConn As SqlConnection = New SqlConnection(My.Settings.NORTHWINDConnectionString1)
        nwindConn.Open()

        ' ... 其他 ADO.NET 数据库访问代码 ... '

        ' 使用预先存在的 ADO.NET 连接创建 DataContext:
        Dim interop_db = New NorthwindDataContext(nwindConn) With {.Log = db.Log}

        Dim orders = From ord In interop_db.Orders _
                     Where ord.Freight > 500D _
                     Select ord

        ObjectDumper.Write(orders)

        nwindConn.Close()
    End Sub

    <Category("ADO.NET 互操作")> _
    <Title("事务互操作")> _
    <Description("此示例使用预先存在的 ADO.NET 连接创建 Northwind  对象，" & _
                 "然后与此对象共享一个 ADO.NET 事务。此事务" & _
                 "既用于通过 ADO.NET 连接执行 SQL 命令，又用于" & _
                 "通过 Northwind 对象提交更改。当事务因违反 CHECK  约束而中止时，" & _
                 "将回滚所有更改，包括" & _
                 "通过 SqlCommand 做出的更改，以及通过" & _
                 "Northwind 对象做出的更改。")> _
    Public Sub LinqToSqlAdo02()
        Dim prodQuery = From prod In db.Products _
                        Where prod.ProductID = 3

        Console.WriteLine("*** BEFORE ***")
        ObjectDumper.Write(prodQuery)


        Console.WriteLine()
        Console.WriteLine("*** INSERT ***")

        ' 创建标准 ADO.NET 连接:
        Dim nwindConn As SqlConnection = New SqlConnection(My.Settings.NORTHWINDConnectionString1)
        nwindConn.Open()

        ' 使用预先存在的 ADO.NET 连接创建 DataContext:
        Dim interop_db As NorthwindDataContext = New NorthwindDataContext(nwindConn)
        interop_db.Log = db.Log

        Dim nwindTxn As SqlTransaction = nwindConn.BeginTransaction()

        Try
            Dim cmd As SqlCommand = New SqlCommand("UPDATE Products SET QuantityPerUnit = 'single item' WHERE ProductID = 3")
            cmd.Connection = nwindConn
            cmd.Transaction = nwindTxn
            cmd.ExecuteNonQuery()

            interop_db.Transaction = nwindTxn

            Dim prod1 = (From prod In interop_db.Products _
                         Where prod.ProductID = 4).First()
            Dim prod2 = (From prod In interop_db.Products _
                         Where prod.ProductID = 5).First()

            prod1.UnitsInStock = prod1.UnitsInStock - 3S
            prod2.UnitsInStock = prod2.UnitsInStock - 5S    ' 错误: 这将使库存量变为负值

            interop_db.SubmitChanges()

            nwindTxn.Commit()
        Catch e As Exception
            ' 如果事务发生错误，则将回滚所有更改，
            ' 包其中包括直接通过 ADO.NET 连接进行的任何更改
            Console.WriteLine(e.Message)
            Console.WriteLine("Error submitting changes... all changes rolled back.")
        End Try

        nwindConn.Close()


        Console.WriteLine()
        Console.WriteLine("*** AFTER ***")
        clearDBCache()
        ObjectDumper.Write(prodQuery)



        Cleanup101()  ' 还原以前的数据库状态
    End Sub


    Private Sub Cleanup101()
        setLogging(False)

        ' 事务失败后将自动回滚数据
    End Sub

    <Category("存储过程")> _
    <Title("标量返回")> _
    <Description("此示例使用存储过程返回在“WA”地区的客户数。")> _
    Public Sub LinqToSqlStoredProc01()
        Dim count = db.Customers_Count_By_Region("WA")

        Console.WriteLine(count)
    End Sub

    <Category("存储过程")> _
    <Title("单一结果集")> _
    <Description("此示例使用映射到 Northwind 数据库中“Customers By City”存储过程" & _
                 "的方法返回来自“London”的客户。" & _
                 "通过将存储过程从服务器资源管理器" & _
                 "拖动到 O/R 设计器(可以通过双击解决方案资源管理器中的" & _
                 ".DBML 文件访问)上即可创建方法。")> _
    Public Sub LinqToSqlStoredProc02()
        Dim custQuery = db.Customers_By_City("London")

        ObjectDumper.Write(custQuery, 0)
    End Sub

    <Category("存储过程")> _
    <Title("多个结果集")> _
    <Description("此示例使用存储过程返回客户“SEVES”及其所有订单。")> _
    Public Sub LinqToSqlStoredProc04()
        Dim result = db.Get_Customer_And_Orders("SEVES")

        Console.WriteLine("********** Customer Result-set ***********")
        Dim customer As IEnumerable(Of Get_Customer_And_OrdersResult) = result
        ObjectDumper.Write(customer)
        Console.WriteLine()

    End Sub

    <Category("存储过程")> _
    <Title("Out 参数")> _
    <Description("此示例使用返回 Out 参数的存储过程。")> _
    Public Sub LinqToSqlStoredProc05()
        Dim totalSales? = 0@

        Dim customerID = "ALFKI"

        ' ByRef 传递 Out 参数，以支持
        ' 参数为 In 或 Out 的情况。在本例中参数仅为
        ' Out。
        db.CustOrderTotal(customerID, totalSales)

        Console.WriteLine("Total Sales for Customer '{0}' = {1:C}", customerID, totalSales)
    End Sub

    <Category("存储过程")> _
    <Title("函数")> _
    <Description("此示例使用映射到 Northwind 数据库中“ProductsUnderThisUnitPrice”函数的" & _
                 "方法返回单价小于 10.00 美元的产品。" & _
                 "通过将数据库函数从服务器资源管理器" & _
                 "拖动到 O/R 设计器(可以通过双击解决方案资源管理器中的" & _
                 ".DBML 文件访问)上即可创建方法。")> _
    Public Sub LinqToSqlStoredProc06()
        Dim cheapProducts = db.ProductsUnderThisUnitPrice(10D)

        ObjectDumper.Write(cheapProducts, 0)
    End Sub

    <Category("存储过程")> _
    <Title("对方法进行查询")> _
    <Description("此示例针对“ProductsUnderThisUnitPrice”方法返回的产品集合" & _
                 "进行查询。该方法是根据" & _
                 "Northwind 数据库中“ProductsUnderThisUnitPrice”函数创建的。")> _
    Public Sub LinqToSqlStoredProc07()
        Dim cheapProducts = From prod In db.ProductsUnderThisUnitPrice(10D) _
                            Where prod.Discontinued = True

        ObjectDumper.Write(cheapProducts, 0)
    End Sub

    <Category("用户定义的函数")> _
    <Title("标量函数 - Select")> _
    <Description("此示例演示在投影中使用用户定义的标量函数。")> _
    Public Sub LinqToSqlUserDefined01()
        Dim catQuery = From category In db.Categories _
                       Select category.CategoryID, _
                              TotalUnitPrice = db.TotalProductUnitPriceByCategory(category.CategoryID)

        ObjectDumper.Write(catQuery)
    End Sub

    <Category("用户定义的函数")> _
    <Title("标量函数 - Where")> _
    <Description("此示例演示在 Where 子句中使用用户定义的标量函数。")> _
    Public Sub LinqToSqlUserDefined02()

        Dim prodQuery = From prod In db.Products _
                        Where prod.UnitPrice = db.MinUnitPriceByCategory(prod.CategoryID)

        ObjectDumper.Write(prodQuery)
    End Sub

    <Category("用户定义的函数")> _
    <Title("表值函数")> _
    <Description("此示例演示从用户定义的表值函数进行选择。")> _
    Public Sub LinqToSqlUserDefined03()

        Dim prodQuery = From p In db.ProductsUnderThisUnitPrice(10.25D) _
                        Where Not p.Discontinued

        ObjectDumper.Write(prodQuery)
    End Sub

    <Category("用户定义的函数")> _
    <Title("表值函数 - Join")> _
    <Description("此示例演示联接用户定义的表值函数的结果。")> _
    Public Sub LinqToSqlUserDefined04()

        Dim q = From category In db.Categories _
                Group Join prod In db.ProductsUnderThisUnitPrice(8.5D) _
                      On category.CategoryID Equals prod.CategoryID _
                Into prods = Group _
                From prod2 In prods _
                Select category.CategoryID, category.CategoryName, _
                       prod2.ProductName, prod2.UnitPrice

        ObjectDumper.Write(q)
    End Sub

    <Category("DataContext 函数")> _
    <Title("CreateDatabase() 和 DeleteDatabase() ")> _
    <Description("此示例使用 CreateDatabase() 基于 Mapping.cs 中的 NewCreateDB 架构创建新数据库，" & _
                 "并使用 DeleteDatabase() 删除这个新创建的数据库。")> _
    Public Sub LinqToSqlDataContext01()

        ' 创建一个临时文件夹以存储新创建的数据库 
        Dim userTempFolder = Environment.GetEnvironmentVariable("SystemDrive") + "\LinqToSqlSamplesTemp"
        Directory.CreateDirectory(userTempFolder)

        Console.WriteLine("********** Create NewCreateDB ***********")
        Dim userMDF = System.IO.Path.Combine(userTempFolder, "NewCreateDB.mdf")
        Dim connStr = "Data Source=.\SQLEXPRESS;AttachDbFilename=" & userMDF & ";Integrated Security=True;Connect Timeout=30;User Instance=True; Integrated Security = SSPI;"
        Dim newDB = New NewCreateDB(connStr)
        Try
            newDB.CreateDatabase()
        Catch ex As SqlException
            Console.WriteLine(ex.Message)
        End Try

        If (newDB.DatabaseExists() AndAlso File.Exists(Path.Combine(userTempFolder, "NewCreateDB.mdf"))) Then
            Console.WriteLine("NewCreateDB is created")
        Else
            Console.WriteLine("Error: NewCreateDB is not successfully created")
        End If


        Console.WriteLine()
        Console.WriteLine("************ Delete NewCreateDB **************")
        newDB.DeleteDatabase()

        If (File.Exists(Path.Combine(userTempFolder, "NewCreateDB.mdf"))) Then
            Console.WriteLine("Error: NewCreateDB is not yet deleted")
        Else
            Console.WriteLine("NewCreateDB is deleted")
        End If
        ' 删除为此测试用例创建的临时文件夹 
        Directory.Delete(userTempFolder)

    End Sub

    <Category("DataContext 函数")> _
    <Title("DatabaseExists()")> _
    <Description("此示例使用 DatabaseExists() 验证数据库是否存在。")> _
    Public Sub LinqToSqlDataContext02()

        Console.WriteLine("*********** Verify Northwind DB exists ***********")
        If (db.DatabaseExists()) Then
            Console.WriteLine("Northwind DB exists")
        Else
            Console.WriteLine("Error: Northwind DB does not exist")

            Console.WriteLine()
            Console.WriteLine("********* Verify NewCreateDB does not exist **********")
            Dim userTempFolder = Environment.GetEnvironmentVariable("Temp")
            Dim userMDF = System.IO.Path.Combine(userTempFolder, "NewCreateDB.mdf")
            Dim newDB = New NewCreateDB(userMDF)

            If (newDB.DatabaseExists()) Then
                Console.WriteLine("Error: NewCreateDB DB exists")
            Else
                Console.WriteLine("NewCreateDB DB does not exist")
            End If
        End If
    End Sub

    <Category("DataContext 函数")> _
    <Title("SubmitChanges()")> _
    <Description("此示例演示必须调用 SubmitChanges() 才能" & _
                 "向实际的数据库提交更新。")> _
    Public Sub LinqToSql1DataContext03()
        Dim customer = db.Customers.First(Function(cust) cust.CustomerID = "ALFKI")

        Console.WriteLine("********** Original Customer CompanyName **********")
        Dim custQuery = From cust In db.Customers _
                        Where cust.CustomerID = "ALFKI" _
                        Select cust.CompanyName

        Console.WriteLine()
        ObjectDumper.Write(custQuery)

        Console.WriteLine()
        Console.WriteLine("*********** Update and call SubmitChanges() **********")

        customer.CompanyName = "VB Programming Firm"
        db.SubmitChanges()

        Console.WriteLine()
        ObjectDumper.Write(custQuery)

        Console.WriteLine()
        Console.WriteLine("*********** Update and did not call SubmitChanges() **********")

        customer.CompanyName = "LinqToSql Programming Firm"

        Console.WriteLine()
        ObjectDumper.Write(custQuery)

        Cleanup122()  ' 还原以前的数据库状态      
    End Sub

    Private Sub Cleanup122()
        setLogging(False)
        Dim cust = db.Customers.First(Function(c) c.CustomerID = "ALFKI")
        cust.CompanyName = "Alfreds Futterkiste"
        db.SubmitChanges()
    End Sub

    <Category("DataContext 函数")> _
    <Title("CreateQuery()")> _
    <Description("此示例使用 CreateQuery() 从表达式创建 IQueryable(Of T)。")> _
    Public Sub LinqToSqlDataContext04()

        Dim custParam = Expression.Parameter(GetType(Customer), "c")
        Dim city = GetType(Customer).GetProperty("City")

        Dim pred = Expression.Lambda(Of Func(Of Customer, Boolean))( _
            Expression.Equal( _
                             Expression.Property(custParam, city), Expression.Constant("Seattle") _
                             ), custParam)

        Dim custs As IQueryable = db.Customers
        Dim expr = Expression.Call(GetType(Queryable), "Where", _
                                   New Type() {custs.ElementType}, custs.Expression, pred)

        Dim custQuery = db.Customers.AsQueryable().Provider.CreateQuery(Of Customer)(expr)

        ObjectDumper.Write(custQuery)
    End Sub

    <Category("DataContext 函数")> _
    <Title("日志")> _
    <Description("此示例使用 Db.Log 关闭和打开数据库的日志记录显示。")> _
    Public Sub LinqToSqlDataContext05()
        Console.WriteLine("**************** Turn off DB Log Display *****************")
        db.Log = Nothing
        Dim londonCustomers = From cust In db.Customers _
                              Where cust.City = "London"

        ObjectDumper.Write(londonCustomers)

        Console.WriteLine()
        Console.WriteLine("**************** Turn on DB Log Display  *****************")

        db.Log = Me.OutputStreamWriter
        ObjectDumper.Write(londonCustomers)


    End Sub

    <Category("高级")> _
    <Title("动态查询 - Select")> _
    <Description("此示例动态生成查询以返回每个客户的联系人姓名。")> _
    Public Sub LinqToSqlAdvanced01()
        Dim param = Expression.Parameter(GetType(Customer), "c")
        Dim selector = Expression.Property(param, GetType(Customer).GetProperty("ContactName"))
        Dim pred = Expression.Lambda(selector, param)

        Dim custs = db.Customers
        Dim expr = Expression.Call(GetType(Queryable), "Select", New Type() {GetType(Customer), GetType(String)}, Expression.Constant(custs), pred)
        Dim query = custs.AsQueryable().Provider.CreateQuery(Of String)(expr)

        Dim cmd = db.GetCommand(query)
        Console.WriteLine("Generated T-SQL:")
        Console.WriteLine(cmd.CommandText)
        Console.WriteLine()


        ObjectDumper.Write(query)
    End Sub

    <Category("高级")> _
    <Title("动态查询 - Where")> _
    <Description("此示例动态生成查询以筛选在伦敦的客户。")> _
    Public Sub LinqToSqlAdvanced02()

        Dim custs = db.Customers
        Dim param = Expression.Parameter(GetType(Customer), "c")
        Dim right = Expression.Constant("London")
        Dim left = Expression.Property(param, GetType(Customer).GetProperty("City"))
        Dim filter = Expression.Equal(left, right)
        Dim pred = Expression.Lambda(filter, param)

        Dim expr = Expression.Call(GetType(Queryable), "Where", New Type() {GetType(Customer)}, Expression.Constant(custs), pred)
        Dim query = db.Customers.AsQueryable().Provider.CreateQuery(Of Customer)(expr)
        ObjectDumper.Write(query)
    End Sub


    <Category("高级")> _
    <Title("动态查询 - OrderBy")> _
    <Description("此示例动态生成查询以筛选在伦敦的客户，" & _
                 " 并按 ContactName 对这些客户进行排序。")> _
    Public Sub LinqToSqlAdvanced03()

        Dim param = Expression.Parameter(GetType(Customer), "c")

        Dim left = Expression.Property(param, GetType(Customer).GetProperty("City"))
        Dim right = Expression.Constant("London")
        Dim filter = Expression.Equal(left, right)
        Dim pred = Expression.Lambda(filter, param)

        Dim custs As IQueryable = db.Customers

        Dim expr = Expression.Call(GetType(Queryable), "Where", _
                                   New Type() {GetType(Customer)}, _
                                   Expression.Constant(custs), pred)

        expr = Expression.Call(GetType(Queryable), "OrderBy", _
                               New Type() {GetType(Customer), GetType(String)}, _
                               custs.Expression, _
                               Expression.Lambda(Expression.Property(param, "ContactName"), param))


        Dim query = db.Customers.AsQueryable().Provider.CreateQuery(Of Customer)(expr)

        ObjectDumper.Write(query)
    End Sub

    <Category("高级")> _
    <Title("动态查询 - Union")> _
    <Description("此示例动态生成联合，以返回客户或雇员任意一方" & _
                 "居住的所有国家/地区的序列。")> _
    Public Sub LinqToSqlAdvanced04()

        Dim custs = db.Customers
        Dim param1 = Expression.Parameter(GetType(Customer), "c")
        Dim left1 = Expression.Property(param1, GetType(Customer).GetProperty("City"))
        Dim pred1 = Expression.Lambda(left1, param1)

        Dim employees = db.Employees
        Dim param2 = Expression.Parameter(GetType(Employee), "e")
        Dim left2 = Expression.Property(param2, GetType(Employee).GetProperty("City"))
        Dim pred2 = Expression.Lambda(left2, param2)

        Dim expr1 = Expression.Call(GetType(Queryable), "Select", New Type() {GetType(Customer), GetType(String)}, Expression.Constant(custs), pred1)
        Dim expr2 = Expression.Call(GetType(Queryable), "Select", New Type() {GetType(Employee), GetType(String)}, Expression.Constant(employees), pred2)

        Dim custQuery1 = db.Customers.AsQueryable().Provider.CreateQuery(Of String)(expr1)
        Dim empQuery1 = db.Employees.AsQueryable().Provider.CreateQuery(Of String)(expr2)

        Dim finalQuery = custQuery1.Union(empQuery1)

        ObjectDumper.Write(finalQuery)
    End Sub

    <Category("高级")> _
    <Title("标识")> _
    <Description("此示例演示如何插入新联系人以及从" & _
                 "数据库检索新分配的 ContactID。")> _
    Public Sub LinqToSqlAdvanced05()

        Console.WriteLine("ContactID is marked as an identity column")
        Dim con = New Contact() With {.CompanyName = "New Era", .Phone = "(123)-456-7890"}

        db.Contacts.InsertOnSubmit(con)
        db.SubmitChanges()

        Console.WriteLine()
        Console.WriteLine("The ContactID of the new record is " & con.ContactID)

        cleanup130(con.ContactID)

    End Sub
    Sub cleanup130(ByVal contactID As Integer)
        setLogging(False)
        Dim con = db.Contacts.Where(Function(c) c.ContactID = contactID).First()
        db.Contacts.DeleteOnSubmit(con)
        db.SubmitChanges()
    End Sub


    <Category("高级")> _
    <Title("嵌套在 FROM 中")> _
    <Description("此示例使用 OrderByDescending 和 Take 返回" & _
                 "最贵的 10 种产品中断货的产品。")> _
    Public Sub LinqToSqlAdvanced06()
        Dim prods = From prod In db.Products.OrderByDescending(Function(p) p.UnitPrice) _
                    Take 10 _
                    Where prod.Discontinued

        ObjectDumper.Write(prods, 0)
    End Sub

    <Category("高级")> _
    <Title("可变/不可变匿名类型")> _
    <Description("" & _
                 "")> _
    Public Sub LinqToSqlAdvanced07()

        '生成不可变的匿名类型(FirstName 和 LastName 将为 ReadOnly 属性)
        Dim empQuery1 = From emp In db.Employees _
                        Where emp.City = "Seattle" _
                        Select emp.FirstName, emp.LastName

        For Each row In empQuery1
            '提取的类型不可变，因此这些行在编译时会出错
            'row.FirstName = "John"
            'row.LastName = "Doe"
        Next

        '生成可变的匿名类型(FirstName 和 LastName 为 Read/Write 属性)
        Dim empQuery2 = From emp In db.Employees _
                        Where emp.City = "Seattle" _
                        Select New With {emp.FirstName, emp.LastName}

        For Each row In empQuery2
            '提取的类型可变，因此这些行将正常运行
            row.FirstName = "John"
            row.LastName = "Doe"
        Next

        '生成部分可变的匿名类型(FirstName 为 ReadOnly 属性，LastName 为 Read/Write 属性)
        Dim empQuery3 = From emp In db.Employees _
                        Where emp.City = "Seattle" _
                        Select New With {Key emp.FirstName, emp.LastName}

        For Each row In empQuery3
            '由于类型为部分可变，因此仅第二行将正常运行；您可以
            '仅写入 LastName 字段
            'row.FirstName = "John"
            row.LastName = "Doe"
        Next

        '生成不可变的匿名类型(FirstName 和 LastName 将为 ReadOnly 属性)
        Dim empQuery4 = From emp In db.Employees _
                        Where emp.City = "Seattle" _
                        Select New With {Key emp.FirstName, Key emp.LastName}

        For Each row In empQuery4
            '提取的类型不可变，因此这些行在编译时会出错
            'row.FirstName = "John"
            'row.LastName = "Doe"
        Next

        Console.WriteLine("By default the Select keyword generates types that are " & _
                          "immutable, however you can use the New With {...} syntax " & _
                          "to generate a mutable anonymous type.  To make individual " & _
                          "fields ReadOnly, use the Key modifier.")
    End Sub

    <Category("视图")> _
    <Title("查询 - 匿名类型")> _
    <Description("此示例使用 Select 和 Where 返回到货城市" & _
                 "为伦敦的发票的序列。")> _
    Public Sub LinqToSqlView01()
        Dim shipToLondon = From inv In db.Invoices _
                           Where inv.ShipCity = "London" _
                           Select inv.OrderID, inv.ProductName, inv.Quantity, inv.CustomerName

        ObjectDumper.Write(shipToLondon, 1)
    End Sub

    <Category("View")> _
    <Title("视图 - 标识映射")> _
    <Description("此示例使用 Select 查询 QuarterlyOrders。")> _
    Public Sub LinqToSqlView02()
        Dim quarterlyOrders = From qo In db.Quarterly_Orders _
                              Select qo

        ObjectDumper.Write(quarterlyOrders, 1)
    End Sub

    <Category("继承")> _
    <Title("简单")> _
    <Description("此示例返回所在城市为伦敦的所有联系人。")> _
    Public Sub LinqToSqlInheritance01()

        Dim cons = From contact In newDB.Contacts _
                   Select contact

        For Each con In cons
            Console.WriteLine("Company name: " & con.CompanyName)
            Console.WriteLine("Phone: " & con.Phone)
            Console.WriteLine("This is a " & con.GetType().ToString)
            Console.WriteLine()
        Next

    End Sub

    <Category("继承")> _
    <Title("TypeOf")> _
    <Description("此示例使用 OfType 返回所有客户联系人。")> _
    Public Sub LinqToSqlInheritance02()

        Dim cons = From contact In newDB.Contacts.OfType(Of CustomerContact)() _
                   Select contact

        ObjectDumper.Write(cons, 0)
    End Sub

    <Category("继承")> _
    <Title("IS")> _
    <Description("此示例使用 IS 返回所有发货方联系人。")> _
    Public Sub LinqToSqlInheritance03()

        Dim cons = From contact In newDB.Contacts _
                   Where TypeOf contact Is ShipperContact _
                   Select contact

        ObjectDumper.Write(cons, 0)
    End Sub


    <Category("继承")> _
    <Title("CType")> _
    <Description("此示例使用 CType 返回 FullContact 或 Nothing。")> _
    Public Sub LinqToSqlInheritance04()
        Dim cons = From contact In newDB.Contacts _
                   Select CType(contact, FullContact)

        ObjectDumper.Write(cons, 0)
    End Sub

    <Category("继承")> _
    <Title("Cast")> _
    <Description("此示例使用强制转换检索居住在伦敦的客户联系人。")> _
    Public Sub LinqToSqlInheritance05()
        Dim cons = From contact In newDB.Contacts _
                   Where contact.ContactType = "Customer" _
                         AndAlso (DirectCast(contact, CustomerContact)).City = "London"

        ObjectDumper.Write(cons, 0)
    End Sub

    <Category("继承")> _
    <Title("UseAsDefault")> _
    <Description("此示例演示未知联系人类型" & _
                 "将被自动转换为默认联系人类型。")> _
    Public Sub LinqToSqlInheritance06()

        Console.WriteLine("***** INSERT Unknown Contact using normal mapping *****")
        Dim contact = New Contact() With {.ContactType = Nothing, _
                                          .CompanyName = "Unknown Company", _
                                          .City = "London", _
                                          .Phone = "333-444-5555"}
        db.Contacts.InsertOnSubmit(contact)
        db.SubmitChanges()

        Console.WriteLine()
        Console.WriteLine("***** Query Unknown Contact using inheritance mapping *****")
        Dim con = (From cont In db.Contacts _
                   Where cont.CompanyName = "Unknown Company" AndAlso _
                         cont.Phone = "333-444-5555").First()

        Console.WriteLine("The base class nwind.BaseContact had been used as default fallback type")
        Console.WriteLine("The discriminator value for con is unknown. So, its type should be " & con.GetType().ToString())

        cleanup140(contact.ContactID)

    End Sub

    Sub cleanup140(ByVal contactID As Integer)
        setLogging(False)
        Dim con = db.Contacts.Where(Function(cont) cont.ContactID = contactID).First()
        db.Contacts.DeleteOnSubmit(con)
        db.SubmitChanges()
    End Sub


    <Category("继承")> _
    <Title("插入新记录")> _
    <Description("此示例演示如何创建新发货方联系人。")> _
    Public Sub LinqToSqlInheritance07()

        Console.WriteLine("****** Before Insert Record ******")
        Dim ShipperContacts = From sc In newDB.Contacts.OfType(Of ShipperContact)() _
                              Where sc.CompanyName = "Northwind Shipper"

        Console.WriteLine()
        Console.WriteLine("There are " & ShipperContacts.Count() & " Shipper Contacts that matched our requirement")

        Dim nsc = New ShipperContact() With {.CompanyName = "Northwind Shipper", _
                                             .Phone = "(123)-456-7890"}
        newDB.Contacts.InsertOnSubmit(nsc)
        newDB.SubmitChanges()

        Console.WriteLine()
        Console.WriteLine("****** After Insert Record ******")
        ShipperContacts = From sc In newDB.Contacts.OfType(Of ShipperContact)() _
                          Where sc.CompanyName = "Northwind Shipper"

        Console.WriteLine()
        Console.WriteLine("There are " & ShipperContacts.Count() & " Shipper Contacts that matched our requirement")
        newDB.Contacts.DeleteOnSubmit(nsc)
        newDB.SubmitChanges()

    End Sub


    <Category("外部映射")> _
    <Title("加载和使用外部映射")> _
    <Description("此示例演示如何创建使用外部 XML 映射源的数据上下文。")> _
    Public Sub LinqToSqlExternal01()

        ' 加载映射源
        Dim path2 = Path.Combine(Application.StartupPath, "..\..\SampleData\NorthwindMapped.map")

        Dim mappingSource As XmlMappingSource = XmlMappingSource.FromXml(File.ReadAllText(path2))


        '请注意，NorthwindMapped.map 文件中的每种类型都包含完全限定
        '名称(即，它“包括”根命名空间)。举例来说，映射文件中的以下
        '元素:
        '    <Type Name="Mapped.AddressSplit">
        '
        '将变为:
        '    <Type Name="SampleQueries.Mapped.AddressSplit">
        '
        '因为 SampleQueries 是为此项目定义的根命名空间

        ' 使用映射源创建上下文
        Dim nw As New Mapped.NorthwindMapped(db.Connection, mappingSource)

        ' 演示外部映射的实体的使用
        Console.WriteLine("****** Externally-mapped entity ******")
        Dim order As Mapped.Order = nw.Orders.First()
        ObjectDumper.Write(order, 1)

        ' 演示外部映射的继承层次结构的使用
        Dim contacts = From c In nw.Contacts _
                       Where TypeOf c Is Mapped.EmployeeContact _
                       Select c

        Console.WriteLine()
        Console.WriteLine("****** Externally-mapped inheritance hierarchy ******")

        For Each contact In contacts
            Console.WriteLine("Company name: {0}", contact.CompanyName)
            Console.WriteLine("Phone: {0}", contact.Phone)
            Console.WriteLine("This is a {0}", contact.GetType())
            Console.WriteLine()
        Next

        ' 演示外部映射的存储过程的使用
        Console.WriteLine()
        Console.WriteLine("****** Externally-mapped stored procedure ******")
        For Each result In nw.CustomerOrderHistory("ALFKI")
            ObjectDumper.Write(result, 0)
        Next

        ' 演示外部映射的用户定义标量函数的使用
        Console.WriteLine()
        Console.WriteLine("****** Externally-mapped scalar UDF ******")
        Dim totals = From c In nw.Categories _
                     Select c.CategoryID, TotalUnitPrice = nw.TotalProductUnitPriceByCategory(c.CategoryID)
        ObjectDumper.Write(totals)

        ' 演示外部映射的用户定义表值函数的使用
        Console.WriteLine()
        Console.WriteLine("****** Externally-mapped table-valued UDF ******")
        Dim products = From p In nw.ProductsUnderThisUnitPrice(9.75D) _
                       Where p.SupplierID = 8 _
                       Select p
        ObjectDumper.Write(products)
    End Sub


    <Category("开放式并发")> _
    <Title("获取冲突信息")> _
    <Description("此示例演示如何检索导致开放式并发异常的更改。")> _
    Public Sub LinqToSqlOptimistic01()
        Console.WriteLine("YOU:  ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~")
        Dim product = db.Products.First(Function(prod) prod.ProductID = 1)
        Console.WriteLine("~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~")
        Console.WriteLine()
        Console.WriteLine("OTHER USER: ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~")

        ' 打开与数据库的第二个连接以模拟
        ' 将要对 Products 表进行更改的另一个用户
        Dim otherUser_db = New NorthwindDataContext(My.Settings.NORTHWINDConnectionString1) With {.Log = db.Log}
        Dim otherUser_product = otherUser_db.Products.First(Function(p) p.ProductID = 1)

        otherUser_product.UnitPrice = 999.99D
        otherUser_product.UnitsOnOrder = 10
        otherUser_db.SubmitChanges()

        Console.WriteLine("~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~")
        Console.WriteLine("YOU (continued): ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~")
        product.UnitPrice = 777.77D

        Dim conflictOccurred = False
        Try
            db.SubmitChanges(ConflictMode.ContinueOnConflict)
        Catch c As ChangeConflictException
            Console.WriteLine("* * * OPTIMISTIC CONCURRENCY EXCEPTION * * *")
            For Each aConflict In db.ChangeConflicts
                Dim prod = CType(aConflict.Object, Product)
                Console.WriteLine("The conflicting product has ProductID " & prod.ProductID)
                Console.WriteLine()
                Console.WriteLine("Conflicting members:")
                Console.WriteLine()
                For Each memConflict In aConflict.MemberConflicts
                    Dim name = memConflict.Member.Name
                    Dim yourUpdate = memConflict.CurrentValue.ToString()
                    Dim original = memConflict.OriginalValue.ToString()
                    Dim theirUpdate = memConflict.DatabaseValue.ToString()
                    If (memConflict.IsModified) Then

                        Console.WriteLine("'{0}' was updated from {1} to {2} while you updated it to {3}", _
                                              name, original, theirUpdate, yourUpdate)
                    Else
                        Console.WriteLine("'{0}' was updated from {1} to {2}, you did not change it.", _
                                                                            name, original, theirUpdate)
                    End If
                    Console.WriteLine()
                Next
                conflictOccurred = True
            Next

            Console.WriteLine()
            If (Not conflictOccurred) Then

                Console.WriteLine("* * * COMMIT SUCCESSFUL * * *")
                Console.WriteLine("Changes to Product 1 saved.")
            End If
            Console.WriteLine("~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ")

            ResetProducts() ' 清理
        End Try
    End Sub

    Private Sub ResetProducts()
        clearDBCache()
        Dim dbClean = New NorthwindDataContext(My.Settings.NORTHWINDConnectionString1)
        Dim prod(4) As Product
        Dim price = New Decimal() {18D, 19D, 10D, 22D}
        For i = 0 To 3
            Dim tmp = i
            prod(i) = dbClean.Products.First(Function(p) p.ProductID = tmp + 1)
            prod(i).UnitPrice = price(i)
        Next
        prod(0).UnitsInStock = 39
        prod(0).UnitsOnOrder = 0
        dbClean.SubmitChanges()
    End Sub

    'OptimisticConcurrencyConflict
    Private Sub WriteConflictDetails(ByVal conflicts As IEnumerable(Of ObjectChangeConflict))
        'OptimisticConcurrencyConflict
        For Each conflict In conflicts
            DescribeConflict(conflict)
        Next
    End Sub

    Private Sub DescribeConflict(ByVal conflict As ObjectChangeConflict)
        Dim prod = DirectCast(conflict.Object, Product)
        Console.WriteLine("Optimistic Concurrency Conflict in product " & prod.ProductID)

        'OptimisticConcurrencyMemberConflict
        For Each memConflict In conflict.MemberConflicts
            Dim name = memConflict.Member.Name
            Dim yourUpdate = memConflict.CurrentValue.ToString()
            Dim original = memConflict.OriginalValue.ToString()
            Dim theirUpdate = memConflict.DatabaseValue.ToString()
            If (memConflict.IsModified) Then
                Console.WriteLine("'{0}' was updated from {1} to {2} while you updated it to {3}", _
                                  name, original, theirUpdate, yourUpdate)
            Else
                Console.WriteLine("'{0}' was updated from {1} to {2}, you did not change it.", _
                    name, original, theirUpdate)
            End If
        Next
    End Sub

    <Category("开放式并发")> _
    <Title("解决冲突: 覆盖当前值")> _
    <Description("此示例演示如何自动解决并发冲突。" & vbNewLine & _
                 "“覆盖当前值”选项将新数据库值写入客户端对象。")> _
    Public Sub LinqToSqlOptimistic02()

        Dim otherUser_db = New NorthwindDataContext(My.Settings.NORTHWINDConnectionString1)
        db.Log = Nothing

        Dim product = db.Products.First(Function(prod) prod.ProductID = 1)
        Console.WriteLine("You retrieve product number 1, it costs " & product.UnitPrice)
        Console.WriteLine("There are {0} units in stock, {1} units on order", product.UnitsInStock, product.UnitsOnOrder)
        Console.WriteLine()

        Console.WriteLine("Another user changes the price to 22.22 and UnitsInStock to 22")
        Dim otherUser_product = otherUser_db.Products.First(Function(prod) prod.ProductID = 1)
        otherUser_product.UnitPrice = 22.22D
        otherUser_product.UnitsInStock = 22
        otherUser_db.SubmitChanges()

        Console.WriteLine("You set the price of product 1 to 1.01 and UnitsOnOrder to 11")
        product.UnitPrice = 1.01D
        product.UnitsOnOrder = 11

        Try
            Console.WriteLine("You submit")
            Console.WriteLine()
            db.SubmitChanges()
        Catch c As ChangeConflictException

            WriteConflictDetails(db.ChangeConflicts)  ' 将更改后的对象/成员写入控制台
            Console.WriteLine()
            Console.WriteLine("Resolve by overwriting current values")

            db.ChangeConflicts.ResolveAll(RefreshMode.OverwriteCurrentValues)
            db.SubmitChanges()
        End Try

        Console.WriteLine()
        Dim dbResult = New NorthwindDataContext(My.Settings.NORTHWINDConnectionString1)
        Dim result = dbResult.Products.First(Function(prod) prod.ProductID = 1)
        Console.WriteLine("Now product 1 has price={0}, UnitsInStock={1}, UnitsOnOrder={2}", _
                          result.UnitPrice, result.UnitsInStock, result.UnitsOnOrder)
        Console.WriteLine()
        ResetProducts() ' 清理
    End Sub

    <Category("开放式并发")> _
    <Title("解决冲突: 保留当前值")> _
    <Description("此示例演示如何自动解决并发冲突。" & vbNewLine & _
                 "“保留当前值”选项将所有内容都更改为此客户端的值。")> _
    Public Sub LinqToSqlOptimistic03()
        Dim otherUser_db = New NorthwindDataContext(My.Settings.NORTHWINDConnectionString1)
        db.Log = Nothing

        Dim prod = db.Products.First(Function(p) p.ProductID = 1)
        Console.WriteLine("You retrieve the product 1, it costs " & prod.UnitPrice)
        Console.WriteLine("There are {0} units in stock, {1} units on order", prod.UnitsInStock, prod.UnitsOnOrder)
        Console.WriteLine()

        Console.WriteLine("Another user changes the price to 22.22 and UnitsInStock to 22")
        Dim otherUser_product = otherUser_db.Products.First(Function(p) p.ProductID = 1)
        otherUser_product.UnitPrice = 22.22D
        otherUser_product.UnitsInStock = 22
        otherUser_db.SubmitChanges()

        Console.WriteLine("You set the price of product 1 to 1.01 and UnitsOnOrder to 11")
        prod.UnitPrice = 1.01D
        prod.UnitsOnOrder = 11

        Try
            Console.WriteLine("You submit")
            Console.WriteLine()
            db.SubmitChanges()
        Catch c As ChangeConflictException
            WriteConflictDetails(db.ChangeConflicts) ' 将更改后的对象/成员写入控制台

            Console.WriteLine()
            Console.WriteLine("Resolve by keeping current values")

            db.ChangeConflicts.ResolveAll(RefreshMode.KeepCurrentValues)
            db.SubmitChanges()
        End Try

        Console.WriteLine()
        Dim dbResult = New NorthwindDataContext(My.Settings.NORTHWINDConnectionString1)
        Dim result = dbResult.Products.First(Function(p) p.ProductID = 1)
        Console.WriteLine("Now product 1 has price={0}, UnitsInStock={1}, UnitsOnOrder={2}", _
                          result.UnitPrice, result.UnitsInStock, result.UnitsOnOrder)
        Console.WriteLine()
        ResetProducts() ' 清理
    End Sub

    <Category("开放式并发")> _
    <Title("解决冲突: 保留更改")> _
    <Description("此示例演示如何自动解决并发冲突。" & vbNewLine & _
                 "“保留更改”选项保留当前用户做出的所有更改，" & _
                 "并合并其他用户做出的更改(如果当前用户未更改相应的字段)。")> _
    Public Sub LinqToSqlOptimistic04()

        Dim otherUser_db = New NorthwindDataContext(My.Settings.NORTHWINDConnectionString1)
        db.Log = Nothing

        Dim prod = db.Products.First(Function(p) p.ProductID = 1)
        Console.WriteLine("You retrieve the product 1, it costs " & prod.UnitPrice)
        Console.WriteLine("There are {0} units in stock, {1} units on order", prod.UnitsInStock, prod.UnitsOnOrder)
        Console.WriteLine()

        Console.WriteLine("Another user changes the price to 22.22 and UnitsInStock to 22")
        Dim otherUser_product = otherUser_db.Products.First(Function(p) p.ProductID = 1)
        otherUser_product.UnitPrice = 22.22D
        otherUser_product.UnitsInStock = 22
        otherUser_db.SubmitChanges()

        Console.WriteLine("You set the price of product 1 to 1.01 and UnitsOnOrder to 11")
        prod.UnitPrice = 1.01D
        prod.UnitsOnOrder = 11S

        Try
            Console.WriteLine("You submit")
            Console.WriteLine()
            db.SubmitChanges()
        Catch c As ChangeConflictException
            WriteConflictDetails(db.ChangeConflicts) '将更改后的对象/成员写入控制台

            Console.WriteLine()
            Console.WriteLine("Resolve by keeping changes")

            db.ChangeConflicts.ResolveAll(RefreshMode.KeepChanges)
            db.SubmitChanges()
        End Try

        Console.WriteLine()

        Dim dbResult = New NorthwindDataContext(My.Settings.NORTHWINDConnectionString1)
        Dim result = dbResult.Products.First(Function(p) p.ProductID = 1)

        Console.WriteLine("Now product 1 has price={0}, UnitsInStock={1}, UnitsOnOrder={2}", _
            result.UnitPrice, result.UnitsInStock, result.UnitsOnOrder)
        Console.WriteLine()
        ResetProducts() ' 清理
    End Sub

    <Category("开放式并发")> _
    <Title("自定义解决规则")> _
    <Description("演示使用 MemberConflict.Resolve 编写自定义解决规则。")> _
    Public Sub LinqToSqlOptimistic05()

        Dim otherUser_db = New NorthwindDataContext(My.Settings.NORTHWINDConnectionString1)
        db.Log = Nothing

        Dim prod = db.Products.First(Function(p) p.ProductID = 1)
        Console.WriteLine("You retrieve the product 1, it costs " & prod.UnitPrice)
        Console.WriteLine("There are {0} units in stock, {1} units on order", prod.UnitsInStock, prod.UnitsOnOrder)
        Console.WriteLine()

        Console.WriteLine("Another user changes the price to 22.22 and UnitsOnOrder to 2")
        Dim otherUser_product = otherUser_db.Products.First(Function(p) p.ProductID = 1)
        otherUser_product.UnitPrice = 22.22D
        otherUser_product.UnitsOnOrder = 2
        otherUser_db.SubmitChanges()

        Console.WriteLine("You set the price of product 1 to 1.01 and UnitsOnOrder to 11")
        prod.UnitPrice = 1.01D
        prod.UnitsOnOrder = 11
        Dim needsSubmit = True
        While needsSubmit
            Try
                Console.WriteLine("You submit")
                Console.WriteLine()
                needsSubmit = False
                db.SubmitChanges()
            Catch c As ChangeConflictException
                needsSubmit = True
                WriteConflictDetails(db.ChangeConflicts) ' 将更改后的对象/成员写入控制台
                Console.WriteLine()
                Console.WriteLine("Resolve by higher price / order")
                For Each conflict In db.ChangeConflicts
                    conflict.Resolve(RefreshMode.KeepChanges)
                    For Each memConflict In conflict.MemberConflicts
                        If (memConflict.Member.Name = "UnitPrice") Then
                            '始终使用最高价格
                            Dim theirPrice = CDec(memConflict.DatabaseValue)
                            Dim yourPrice = CDec(memConflict.CurrentValue)
                            memConflict.Resolve(Math.Max(theirPrice, yourPrice))
                        ElseIf (memConflict.Member.Name = "UnitsOnOrder") Then
                            '始终使用金额更高的订单
                            Dim theirOrder = CShort(memConflict.DatabaseValue)
                            Dim yourOrder = CShort(memConflict.CurrentValue)
                            memConflict.Resolve(Math.Max(theirOrder, yourOrder))
                        End If
                    Next
                Next
            End Try
        End While
        Dim dbResult = New NorthwindDataContext(My.Settings.NORTHWINDConnectionString1)
        Dim result = dbResult.Products.First(Function(p) p.ProductID = 1)
        Console.WriteLine("Now product 1 has price={0}, UnitsOnOrder={1}", _
                          result.UnitPrice, result.UnitsOnOrder)
        Console.WriteLine()
        ResetProducts() '清理
    End Sub

    <Category("开放式并发")> _
    <Title("Submit 带 FailOnFirstConflict")> _
    <Description("检测到第一个冲突时，Submit(FailOnFirstConflict) 即引发开放式并发异常。" & vbNewLine & _
                 "每次只能处理一个异常，因此您对每个冲突都必须提交。")> _
    Public Sub LinqToSqlOptimistic06()

        db.Log = Nothing
        Dim otherUser_db = New NorthwindDataContext(My.Settings.NORTHWINDConnectionString1)

        '您加载 3 种产品
        Dim prod() = db.Products.OrderBy(Function(p) p.ProductID).Take(3).ToArray()
        For i = 0 To 2
            Console.WriteLine("You retrieve the product {0}, it costs {1}", i + 1, prod(i).UnitPrice)
        Next
        '其他用户更改这些产品
        Dim otherUserProd() = otherUser_db.Products.OrderBy(Function(p) p.ProductID).Take(3).ToArray()
        For i = 0 To 2
            Dim otherPrice = (i + 1) * 111.11D
            Console.WriteLine("Other user changes the price of product {0} to {1}", i + 1, otherPrice)
            otherUserProd(i).UnitPrice = otherPrice
        Next
        otherUser_db.SubmitChanges()
        Console.WriteLine("Other user submitted changes")

        '您更改自己加载的产品
        For i = 0 To 2
            Dim yourPrice = (i + 1) * 1.01D
            Console.WriteLine("You set the price of product {0} to {1}", i + 1, yourPrice)
            prod(i).UnitPrice = yourPrice
        Next

        ' 提交
        Dim needsSubmit = True
        While needsSubmit
            Try
                Console.WriteLine("======= You submit with FailOnFirstConflict =======")
                needsSubmit = False
                db.SubmitChanges(ConflictMode.FailOnFirstConflict)
            Catch c As ChangeConflictException
                For Each conflict In db.ChangeConflicts

                    DescribeConflict(conflict) '将更改写入控制台
                    Console.WriteLine("Resolve conflict with KeepCurrentValues")
                    conflict.Resolve(RefreshMode.KeepCurrentValues)
                Next
                needsSubmit = True
            End Try
        End While

        Dim dbResult = New NorthwindDataContext(My.Settings.NORTHWINDConnectionString1)

        For i = 0 To 2
            'lambda 表达式中将使用变量，因此创建一个临时变量
            Dim tmp = i
            Dim result = dbResult.Products.First(Function(p) p.ProductID = tmp + 1)
            Console.WriteLine("Now the product {0} has price {1}", i + 1, result.UnitPrice)
        Next
        ResetProducts() '清理
    End Sub

    <Category("开放式并发")> _
    <Title("Submit 带 ContinueOnConflict")> _
    <Description("Submit(ContinueOnConflict) 收集所有并发冲突，并在检测到最后一个冲突时引发异常。\r\n" & _
                 "所有冲突都在一个 Catch 语句中处理。" & vbNewLine & _
                 "此次更新之前，也有可能其他用户更新过相同的对象，因此有可能引发另一起开放式并发异常，需要再次处理。")> _
    Public Sub LinqToSqlOptimistic07()
        db.Log = Nothing
        Dim otherUser_db = New NorthwindDataContext(My.Settings.NORTHWINDConnectionString1)

        ' 您加载 3 种产品
        Dim prod() = db.Products.OrderBy(Function(p) p.ProductID).Take(3).ToArray()
        For i = 0 To 2
            Console.WriteLine("You retrieve the product {0}, it costs {1}", i + 1, prod(i).UnitPrice)
        Next

        ' 其他用户更改这些产品
        Dim otherUserProd() = otherUser_db.Products.OrderBy(Function(p) p.ProductID).Take(3).ToArray()
        For i = 0 To 2
            Dim otherPrice = (i + 1) * 111.11D
            Console.WriteLine("Other user changes the price of product {0} to {1}", i + 1, otherPrice)
            otherUserProd(i).UnitPrice = otherPrice
        Next
        otherUser_db.SubmitChanges()
        Console.WriteLine("Other user submitted changes")

        ' 您更改自己加载的产品
        For i = 0 To 2
            Dim yourPrice = (i + 1) * 1.01D
            Console.WriteLine("You set the price of product {0} to {1}", i + 1, yourPrice)
            prod(i).UnitPrice = yourPrice
        Next

        ' 提交
        Dim needsSubmit = True
        While needsSubmit

            Try
                Console.WriteLine("======= You submit with ContinueOnConflict =======")
                needsSubmit = False
                db.SubmitChanges(ConflictMode.ContinueOnConflict)

            Catch c As ChangeConflictException

                For Each conflict In db.ChangeConflicts
                    DescribeConflict(conflict) ' 将更改写入控制台
                    Console.WriteLine("Resolve conflict with KeepCurrentValues")
                    conflict.Resolve(RefreshMode.KeepCurrentValues)
                Next

                needsSubmit = True
            End Try
        End While

        Dim dbResult = New NorthwindDataContext(My.Settings.NORTHWINDConnectionString1)
        For i = 0 To 2
            Dim tmp = i
            Dim result = dbResult.Products.First(Function(p) p.ProductID = tmp + 1)
            Console.WriteLine("Now the product {0} has price {1}", i + 1, result.UnitPrice)
        Next

        ResetProducts() '清理
    End Sub

    <Category("扩展性分部方法")> _
    <Title("带有 OnValidate 的 Update")> _
    <Description("此示例重写 Order 类的 OnValidate 分部方法。在更新某个定单时，它会验证 " & _
                 "ShipVia 不能大于 100，否则将引发异常并且不会将更新发送到数据库。")> _
    Public Sub LinqToSqlExtensibility01()

        Dim order = (From o In db.Orders _
                     Where o.OrderID = 10355).First()

        ObjectDumper.Write(order)
        Console.WriteLine()

        Console.WriteLine("***** Update Order to set ShipVia to 120 and submit changes ******")
        Console.WriteLine()

        order.ShipVia = 120
        Try
            db.SubmitChanges()
        Catch e As Exception
            Console.WriteLine("****** Catch exception throw by OnValidate() ******")
            Console.WriteLine(e.Message)
        End Try

        Console.WriteLine()
        Console.WriteLine("****** verify that order's ShipVia didn't get changed in db. ******")
        Dim db2 As New NorthwindDataContext(My.Settings.NORTHWINDConnectionString1)
        Dim order2 = (From o In db2.Orders _
                      Where o.OrderID = 10355).First()

        ObjectDumper.Write(order2)
    End Sub


    Public Overrides Sub HandleException(ByVal e As Exception)
        Console.WriteLine("Unable to connect to Northwind database: " & My.Settings.NORTHWINDConnectionString1)
        Console.WriteLine("Check the connection string in the Application Settings or try restarting SQL Server.")
        Console.WriteLine()
        Console.WriteLine("If the problem persists, see the Troubleshooting section of the Readme for tips.")
        Console.WriteLine()
        Console.WriteLine()
        Console.WriteLine("Exception:")
        MyBase.HandleException(e)
    End Sub

    Public Overrides Sub InitSample()
        clearDBCache()
        setLogging(True)
    End Sub

    Public Sub clearDBCache()
        ' 创建一个新的 Northwind 对象，以空对象缓存从头开始
        ' 新 Northwind 对象将重复利用活动的 ADO.NET 连接

        Dim oldLog As TextWriter

        If (db Is Nothing) Then
            oldLog = Nothing
        Else
            oldLog = db.Log
        End If

        db = New NorthwindDataContext() With {.Log = oldLog}
        newDB = New NorthwindInheritance(My.Settings.NORTHWINDConnectionString1) With {.Log = oldLog}
    End Sub

    Public Sub setLogging(ByVal dologging As Boolean)
        If dologging Then
            db.Log = Me.OutputStreamWriter
            'newDB.Log = Me.OutputStreamWriter
        Else
            db.Log = Nothing
            'newDB.Log = Nothing
        End If
    End Sub

    Public Class Name
        Public FirstName As String
        Public LastName As String
    End Class

End Class


Partial Public Class NewCreateDB : Inherits DataContext

    Public Persons As Table(Of Person)

    Public Sub New(ByVal connection As String)
        MyBase.New(connection)
    End Sub

    Public Sub New(ByVal connection As System.Data.IDbConnection)
        MyBase.New(connection)
    End Sub

End Class

<Table(Name:="_Person")> _
Partial Public Class Person : Implements System.ComponentModel.INotifyPropertyChanged

    Private _PersonID As Integer

    Private _PersonName As String

    Private _Age As Nullable(Of Integer)

    Public Sub New()

    End Sub

    <Column(Storage:="_PersonID", DbType:="INT", IsPrimaryKey:=True)> _
    Public Property PersonID() As Integer
        Get
            Return Me._PersonID
        End Get
        Set(ByVal value As Integer)
            If (Me._PersonID <> value) Then
                Me.OnPropertyChanged("PersonID")
                Me._PersonID = value
                Me.OnPropertyChanged("PersonID")
            End If
        End Set
    End Property

    <Column(Storage:="_PersonName", DbType:="NVarChar(30)")> _
    Public Property PersonName() As String
        Get
            Return Me._PersonName
        End Get
        Set(ByVal value As String)
            If (Me._PersonName <> value) Then
                Me.OnPropertyChanged("PersonName")
                Me._PersonName = value
                Me.OnPropertyChanged("PersonName")
            End If
        End Set
    End Property

    <Column(Storage:="_Age", DbType:="INT")> _
    Public Property Age() As Nullable(Of Integer)
        Get
            Return Me._Age
        End Get
        Set(ByVal value As Nullable(Of Integer))
            If (Me._Age <> value) Then
                Me.OnPropertyChanged("Age")
                Me._Age = value
                Me.OnPropertyChanged("Age")
            End If
        End Set
    End Property

    Public Event PropertyChanged As System.ComponentModel.PropertyChangedEventHandler Implements System.ComponentModel.INotifyPropertyChanged.PropertyChanged

    Protected Overridable Sub OnPropertyChanged(ByVal PropertyName As String)
        RaiseEvent PropertyChanged(Me, New System.ComponentModel.PropertyChangedEventArgs(PropertyName))
    End Sub
End Class

Partial Public Class NorthwindInheritance : Inherits DataContext
    Public Contacts As Table(Of BaseContact)

    Public Sub New(ByVal connection As String)
        MyBase.New(connection)
    End Sub

    Public Sub New(ByVal connection As System.Data.IDbConnection)
        MyBase.New(connection)
    End Sub

    '
    '                  Contact
    '                 /      \
    '       ShipperContact   FullContact (abstract)
    '                       /     |     \
    '               Customer   Employee  Supplier
    '               Contact    Contact   Contact
    '
    <Table(Name:="Contacts")> _
    <InheritanceMapping(Code:="Unknown", Type:=GetType(BaseContact), IsDefault:=True)> _
    <InheritanceMapping(Code:="Customer", Type:=GetType(CustomerContact))> _
    <InheritanceMapping(Code:="Supplier", Type:=GetType(SupplierContact))> _
    <InheritanceMapping(Code:="Employee", Type:=GetType(EmployeeContact))> _
    <InheritanceMapping(Code:="Shipper", Type:=GetType(ShipperContact))> _
    Public Class BaseContact
        <Column(DbType:="INT NOT NULL", IsPrimaryKey:=True, IsDbGenerated:=True)> _
        Public ContactID As Integer

        <Column(DbType:="NVARCHAR(50)", IsDiscriminator:=True)> _
        Public ContactType As String

        <Column(DbType:="NVARCHAR(40)")> _
        Public CompanyName As String

        <Column(DbType:="NVARCHAR(24)")> _
        Public Phone As String
    End Class

    Public Class ShipperContact : Inherits BaseContact
    End Class

    Public MustInherit Class FullContact : Inherits BaseContact
        <Column(DbType:="NVARCHAR(40)")> _
        Public ContactName As String

        <Column(DbType:="NVARCHAR(30)")> _
        Public ContactTitle As String

        <Column(DbType:="NVARCHAR(60)")> _
        Public Address As String

        <Column(DbType:="NVARCHAR(15)")> _
        Public City As String

        <Column(DbType:="NVARCHAR(15)")> _
        Public Region As String

        <Column(DbType:="NVARCHAR(10)")> _
        Public PostalCode As String

        <Column(DbType:="NVARCHAR(15)")> _
        Public Country As String

        <Column(DbType:="NVARCHAR(24)")> _
        Public Fax As String
    End Class

    Public Class SupplierContact : Inherits FullContact
        <Column(DbType:="NTEXT")> _
        Public HomePage As String
    End Class

    Public Class CustomerContact : Inherits FullContact
    End Class


    Public Class EmployeeContact : Inherits FullContact
        <Column(DbType:="NVARCHAR(4)")> _
        Public Extension As String

        <Column(DbType:="NVARCHAR(255)")> _
        Public PhotoPath As String

        <Column(DbType:="IMAGE")> _
        Public Photo As Byte()
    End Class
End Class